'Skrip to backup VHDX disks.
'The skript will automatically create a Volume Shadow (https://en.wikipedia.org/wiki/Shadow_Copy) and create a symbolic link to it, then copy the disks through the symbolic link.
'Important: DO NOT KEEP SNAPSHOTS OF YOUR VHDX DISK
'-------------------------------
'Useful VSS commands:
'vssadmin list shadows 
'vssadmin delete shadows /all
'vssadmin delete shadows /Shadow={d759030e-9915-41b4-a291-fa5022ba9724}
'List symlinks on D
'dir d:\ /s /a:l
'Create Symlink:
'mklink /d d:\$vss$
'-------------------------------

'1.0	First release 08Aug2019 
'1.1    Modify function funcGetFreeSpace to allow UNC paths and drive letters - 19Feb2020
'1.2    Modify the time stamp inside of the logfile to be independant from regional settings - 26Feb2020

option explicit
dim  strConfVSSPath,boolConfVSSAutomaticCreate,boolConfVSSCheck,boolVSSCheck,strObjectID,strShadowID,intMinFreeDiskSpaceGB,intDataRetention
dim  boolConfWriteLogfile,strConfLogPath,myLoop,strSourcePath,strTargetPath,iSourceFilesCnt,intTotSize,intTotMachines,boolErrorOccured,strVSSDevice,boolRemoveVSSOnExit
redim preserve arrFilterString(0)
redim preserve arrFilterType(0)
redim preserve aSourceFiles(0)
redim preserve aSourceSize(0)

'***BEGIN***
checkInterpreter  '*** Check to see if this script does not run with cscript ***
getConfig         '*** Read hvbackup.xml ***
subCheckVSS       '*** Check and optional create VSS

if boolConfVSSCheck  then  'Final VSS check if configured
 if boolVSSCheck=true then
  syslog "[OK] All VSS checks passed!"
 End IF
End IF

if ucase(funcFilterDriveLetter(strSourcePath)) <> ucase(funcFilterDriveLetter(strConfVSSPath)) then 
 syslog "[FATAL] VSS Drive must be the same as the source"
 wscript.quit
End if

Call subGetSourceFiles(strSourcePath) 'Get all vhdx files

'Get total numbers of VHDX
For myLoop = 1 To UBound(aSourceFiles)
 syslog myLoop & ": " & aSourceFiles(myLoop) & " - (" & aSourceSize(myLoop) & " MB)"
 intTotSize=intTotSize+aSourceSize(myLoop)
 intTotMachines=intTotMachines+1
Next
syslog "---- Total VHDX: " & intTotMachines & " / TotalSize:" & int(intTotSize/1024) & " GB"

if (int(funcGetFreeSpace(strTargetPath)/1024) - int(intTotSize/1024)) < intMinFreeDiskSpaceGB  then  '1.1
 syslog "[WARN] less disk space (" & "Available:" & int(funcGetFreeSpace(strTargetPath)/1024) & " / Required: " & int(intTotSize/1024) & " / MinFree: " & intMinFreeDiskSpaceGB & ")"
Else 
 syslog "[INF] disk space (" & "Available:" & int(funcGetFreeSpace(strTargetPath)/1024) & "GB / Required:" & int(intTotSize/1024) & "GB / Want min. free:" & intMinFreeDiskSpaceGB & "GB)"
End If

subCopyVHD  'Copy all my marked VHDX Disks

If boolRemoveVSSOnExit = 1 then
 subDeleteVSS(funcTestSymLink(strConfVSSPath))
 subDeleteSymLink(strConfVSSPath)
End if

if boolErrorOccured then 
 syslog "----- Errors occured while processing"
else
 syslog "----- Job finished"
end if	 




'********************************************************
'********************************************************
sub subCheckVSS
 'Check VSS, optional create the volume
 Dim strVSSDevice
 if boolConfVSSCheck  then 'Check SymLink and VSS Settings
  syslog "---- Check/Test VSS Link " & strConfVSSPath
  if funcCheckForSymLink(strConfVSSPath) then
   strVSSDevice=funcTestSymLink(strConfVSSPath)
   if len(strVSSDevice)>0 then
    syslog "[OK] Symlink ("& strConfVSSPath &") is present and is linking to: " & strVSSDevice
    'Check VSS
    if funcGetVSSDetails(strVSSDevice) then
     syslog "[OK] Device object ( " & strVSSDevice &  " ) exist"
     boolVSSCheck=true
    else  
     syslog "[INFO] Device object ( " & strVSSDevice &  " ) does not exist"
     boolVSSCheck=false
    end if
   End if 
   if not boolVSSCheck then
    if boolConfVSSAutomaticCreate then 'Delete Symlink then Create VSS then Create Symlink
     syslog "---- Delete SymLink: " & strConfVSSPath
     subDeleteSymLink(strConfVSSPath)
     syslog "---- Now creating VSS"
     strShadowID=funcCreateVSS(strConfVSSPath) 'Create VSS and return shadowID
     if strShadowID = "0" then
      syslog "[FATAL] Cannot create VSS, abort!"
     else 
      syslog "[OK] Created VSS with Shadow Copy ID:" &  strShadowID
      strObjectID=funcGetObjectID(strShadowID) 'Get DeviceID    
      if strObjectID <> "0" then 'No DeviceID given
       syslog "---- Creating SymLink for ObjectID: " &  strObjectID
       call subCreateSymLink(strConfVSSPath,strObjectID)
      else
       syslog "[FATAL] Cannot get VSS ObjectID, abort!"
      End if	   'if strObjectID <> 0 then
     End IF     'if strShadowID = "0" then
    Else    'if boolConfVSSAutomaticCreate then 
     syslog "[FAIL] VSS check failed - abort!"
    End if  'if boolConfVSSAutomaticCreate then 
   End if   'if not boolVSSCheck then
  Else      'if funcCheckForSymLink(strConfVSSPath) -- 'Symlink is not present at all!!
   if boolConfVSSAutomaticCreate then 	'Try to create VSS
    syslog "[WARN] Symlink " & strConfVSSPath & " does not exist"
    strShadowID=funcCreateVSS(strConfVSSPath) 
    'syslog strObjectID
    if strShadowID = "0" then
     syslog "[FATAL] Cannot create VSS, are you administrator?"
    Else 
     syslog "[OK] Created VSS with Shadow Copy ID:" &  strShadowID
     'Get DeviceID
     strObjectID=funcGetObjectID(strShadowID)     
     if strObjectID <> "0" then
      'syslog strObjectID 
      syslog "---- Creating SymLink for ObjectID: " &  strObjectID
      call subCreateSymLink(strConfVSSPath,strObjectID)
      boolVSSCheck=true
     else
      syslog "[Fatal] Cannot get VSS ObjectID, abort!"
     end if	 
    end if	 'if strShadowID = "0" then
   Else     'if boolConfVSSAutomaticCreate then 	 
    syslog "[Fatal] Cannot get VSS ObjectID, abort!"
   End if 	 'if boolConfVSSAutomaticCreate then 	
  End if    'if funcCheckForSymLink(strConfVSSPath)
 Else       'if boolConfVSSCheck then 
  syslog "[SKIP] Symlink check" 
 End if     'if boolConfVSSCheck then
End sub

sub getConfig
 'Read my xml configuration, need components from https://www.chilkatsoft.com/downloads_ActiveX.asp
 dim strSuccess,objXML,numProfile,iCnt,i,objArgs,strConfig,objFso
 set objXML = CreateObject("Chilkat_9_5_0.Xml") 'The XML Object is free and does not require a license
 Set objArgs = Wscript.Arguments
 set objFso = CreateObject("Scripting.FileSystemObject")
 If objArgs.Count =1 then
  'Try to use the first arg as path to configuration
  strConfig=objArgs(0) 
 End If	
 
 if objFso.FileExists(strConfig) then 
  'wscript.echo "Load: " & strConfig
 Else 'Try the default
  if objFso.FileExists("hvbackup.xml") then 
   strConfig="hvbackup.xml"
   'wscript.echo "Load: " & strConfig
  Else
   'No configuration found	
   If len(strConfig)>1 then
    Wscript.echo "Configuration does not exist: "  & strConfig
    wscript.quit
   Else
    wscript.echo "Configuration does not exist:  hvbackup.xml " 
    wscript.quit
   End if  
  End If 
 End if 
 
 strSuccess = objXML.LoadXmlFile(strConfig)
 If (strSuccess <> 1) Then
  syslog (objXML.LastErrorText)
  WScript.Quit
 End If

 strConfVSSPath=objXML.GetChildContent("Setup|VSSPath") 
 strConfVSSPath=funcCutTrailingBackslash(strConfVSSPath)
 boolConfVSSCheck=objXML.GetChildContent("Setup|VSSCheck") 
 boolConfVSSAutomaticCreate=objXML.GetChildContent("Setup|VSSAutomaticCreate") 
 boolConfWriteLogfile=objXML.GetChildContent("Setup|WriteLogfile") 
 strConfLogPath=objXML.GetChildContent("Setup|LogPath") 
 strSourcePath=funcAddTrailingBackslash(objXML.GetChildContent("Setup|SourcePath") )
 strTargetPath=funcAddTrailingBackslash(objXML.GetChildContent("Setup|TargetPath") )
 intMinFreeDiskSpaceGB=int(objXML.GetChildContent("Setup|MinFreeDiskSpaceGB")) 	
 intDataRetention=int(objXML.GetChildContent("Setup|DataRetention")) 	
 boolRemoveVSSOnExit=int(objXML.GetChildContent("Setup|RemoveVSSOnExit")) 	
 
 if boolConfWriteLogfile = 1 then 
  subCreateDirs(strConfLogPath)
 End if 

 numProfile = objXML.NumChildrenAt("Filters")
 'wscript.echo("numProfile = " & numProfile)

 iCnt = 0

 Do While  iCnt < numProfile
    objXML.I = iCnt
    redim preserve arrFilterString(iCnt)
    redim preserve arrFilterType(iCnt)
    arrFilterString(iCnt)=funcSkipVHDExtension(objXML.GetChildContent("Filters|Filter[iCnt]|String"))
    arrFilterType(iCnt)=objXML.GetChildContent("Filters|Filter[iCnt]|Type")
    iCnt = iCnt + 1
 Loop
 set objXML = Nothing
 set objArgs = Nothing
 set objFso = Nothing

End Sub  'getConfig


sub checkInterpreter
 'Checkif we run with cscript, abort if not
 If InStr(1, WScript.FullName, "wscript", vbTextCompare) Then 'Check if wscript is running, quit if
  syslog "Please run this script using cscript"
  wscript.quit
 end if	
End sub

function funcCheckForSymLink(strPath) 
 'Check if the symlink is present, 
 'sample, assume that strPath=c:\$vss\ 
 'output: " 27.04.2019  18:24    <SYMLINKD>     $vss$ [\\?\GLOBALROOT\Device\HarddiskVolumeShadowCopy9]"
 'the above sample would return true then
 funcCheckForSymLink=0
 Dim ObjExec,objShell,strFromProc,strTmpPath
 Set objShell = WScript.CreateObject("WScript.Shell")
 Set ObjExec = objShell.Exec("cmd.exe /c dir " & strPath & "* /a:l")
 Do
  strFromProc = ObjExec.StdOut.ReadLine()
  If InStr(1, strFromProc, "<SYMLINKD>", vbTextCompare) Then 'Check if <SYMLINKD> appears in the output
   strTmpPath=funcSkipDriveLetter(strPath)
   If InStr(1, strFromProc, strTmpPath, vbTextCompare) Then 'Check if 'strPath' appears in the output
    funcCheckForSymLink=1
    exit do	
   End if
  End if
 Loop While Not ObjExec.Stdout.atEndOfStream
 Set objShell = nothing
 Set ObjExec = nothing
End Function


function funcTestSymLink(strPath) 
 'run cmd.exe /i .. /a:l  -  get the symlimk given by strPath, then return the linked path
 Dim ObjExec,objShell,strFromProc,myloop,char,boolStartTrigger,boolStopTrigger,strVSSlink
 Set objShell = WScript.CreateObject("WScript.Shell")
 Set ObjExec = objShell.Exec("cmd.exe /c dir " & strPath & "* /a:l")
 Do
  strFromProc = ObjExec.StdOut.ReadLine()
  for myloop = 1 to len(strFromProc)
   char = Mid(strFromProc,myloop,1)
   if  char = "]" then 
    boolStopTrigger=true
   end if
   if boolStartTrigger and not boolStopTrigger then
    strVSSlink=strVSSlink & char
   end if
   if  char = "[" then 
    boolStartTrigger=true
    'syslog char  
   end if
  next
 Loop While Not ObjExec.Stdout.atEndOfStream
' syslog "VSS Link = " & strVSSlink
 if len(strVSSlink)>0 then
  funcTestSymLink=strVSSlink
 else
  funcTestSymLink=""
 end if
 Set objShell = nothing
 Set ObjExec = nothing
End Function


function funcGetVSSDetails(strDevice)
 dim strComputer,objWMIService,colItems,objItem
 ''https://www.activexperts.com/admin/vbscript-collection/computermanagement/shadow/
 strComputer = "."
 Set objWMIService = GetObject("winmgmts:\\" & strComputer & "\root\cimv2")
 Set colItems = objWMIService.ExecQuery("Select * from Win32_ShadowCopy ")
 For Each objItem in colItems
 if objItem.DeviceObject = strDevice then
    funcGetVSSDetails = true
  end if
 Next
End function

sub subDeleteSymLink(strLink)
 Dim ObjExec,strFromProc,objShell
 Set objShell = WScript.CreateObject("WScript.Shell")
 Set ObjExec = objShell.Exec("cmd.exe /c rmdir " & strLink)
 Set objShell = nothing
 'syslog "cmd.exe /c rmdir " & strLink
End Sub


Function funcCutTrailingBackslash(strPath)
 'Simply cut a trailing backslash if exist, sample: "c:\test\" would return "c:\test"
 dim strtmp
 if mid(strPath,len(strPath),1) = "\" then
  strtmp=mid(strPath,1,len(strPath)-1)
  funcCutTrailingBackslash=strtmp
 Else
  funcCutTrailingBackslash=strPath
 End if
End Function 

Function funcAddTrailingBackslash(strPath)
 'Simply add a trailing backslash if not exist, sample: "c:\test" would return "c:\test\"
 if mid(strPath,len(strPath),1) =  "\" then
  funcAddTrailingBackslash=strPath
 Else
  funcAddTrailingBackslash=strPath & "\"
 End if
End Function 


Function funcSkipDriveLetter(strPath)
 'Simply cut the drive letter with colom and backslash if exist, sample: "c:\test" would return "test"
 dim strtmp 	
 if mid(strPath,2,2) = ":\" then 'assume that this is a drive letter
  'syslog "Ret " & mid(strPath,4,len(strPath))
  strtmp=mid(strPath,4,len(strPath))
  funcSkipDriveLetter=strtmp
 Else
  funcSkipDriveLetter=strPath
 End if
End Function 

Function funcFilterDriveLetter(strPath)
 'Filter the string to obtain the driev letter, sample: "c:\test" would return "c:\"
 dim strtmp 	
 if mid(strPath,2,2) = ":\" then 'assume that this is a drive letter
  'syslog "Ret " & mid(strPath,4,len(strPath))
  strtmp=mid(strPath,1,3)
  funcFilterDriveLetter=strtmp
 Else
  funcFilterDriveLetter=false
 End if
End Function 

Function funcSkipVHDExtension(strPath)
 'Filter the extension of vhx, sample: "c:\test.vhdx" would return "c:\test"
 dim strtmp 	
 if len(strPath) >= 5 then 
  if ucase(mid(strPath,len(strPath)-4,len(strPath)))  = ".VHDX" then 'assume that this is a drive letter
   strtmp=mid(strPath,1,len(strPath)-5)
   funcSkipVHDExtension=strtmp
  Else
   funcSkipVHDExtension=strPath
  End if
 Else
   funcSkipVHDExtension=strPath
 End If	 
End Function 




Function funcCreateVSS(strPath) 
 'Create VSS by strPath, we extract the drive letter from there. Retrun 0 in case of errors
 dim strVolume,strContext,strComputer,strResult,objWMIService,objShadowStorage,strShadowID
 strVolume = funcFilterDriveLetter(strPath)
 strContext = "ClientAccessible"
 strComputer = "."
 Set objWMIService = GetObject("winmgmts:\\" & strComputer & "\root\cimv2")
 Set objShadowStorage = objWMIService.Get("Win32_ShadowCopy")
 strResult = objShadowStorage.Create(strVolume, strContext, strShadowID)
 if strResult <> 0 then
 	funcCreateVSS=0
 else 
 	funcCreateVSS=strShadowID
 end if	
 'syslog strShadowID
 'syslog strResult
 Set objWMIService = Nothing
 Set objShadowStorage = Nothing
End Function

sub subCreateSymLink(strPath,strObjID)
' Execute program and 
' Read the output into a variable line by line
 Dim ObjExec,objShell
 'syslog strObjID
 'syslog strPath
 Set objShell = WScript.CreateObject("WScript.Shell")
 Set ObjExec = objShell.Exec("cmd.exe /c mklink /d " & strPath & " " & strObjID)
 Set objShell = Nothing
 Set ObjExec = Nothing
 
End Sub


Function getDate
 'Return the the date in the format of yyyymmdd including leading zeros
 dim strDate
 strDate = DatePart("yyyy",Date)  _
        & Right("0" & DatePart("m",Date), 2)  _
        & Right("0" & DatePart("d",Date), 2) 
 getDate = strDate        
End Function

Sub syslog(sText)
 'On Error Resume Next
 Dim lfso, lf
 wscript.echo sText
 Set lfso = CreateObject("Scripting.FileSystemObject")
 Set lf = lfso.OpenTextFile(strConfLogPath & "\" & getDate & "-hvbackup.log", 8, True)
 lf.Write getLogDate & " " & sText & vbCrLf
 lf.Close
 Set lf = Nothing
 Set lfso = Nothing
End Sub


function funcGetObjectID(strID) 
 'Return the CSS Object ID by its Shadow ID
 dim strComputer,objWMIService,colItems,objItem
 strComputer = "."
 Set objWMIService = GetObject("winmgmts:\\" & strComputer & "\root\cimv2")
 Set colItems = objWMIService.ExecQuery("Select * from Win32_ShadowCopy where ID ='" & strID & "'")
 funcGetObjectID=0
 For Each objItem in colItems
  funcGetObjectID=objItem.DeviceObject
 Next
 Set objWMIService = Nothing
 Set colItems = Nothing
End Function

Sub subGetSourceFiles(strFolder) 
 'This function makes a recursive call into each subdirectory and collects the name of VHDX files, return the array aSourceFiles
 Dim objFileSys,objFolder,objFile,objSubFolder,strExtension,mysubLoop,boolMark
 strExtension="VHDX"
 Set objFileSys = CreateObject("Scripting.FileSystemObject")
 Set objFolder = objFileSys.GetFolder(strFolder)
 For Each objFile in objFolder.files
 boolMark=false
  if ucase(objFileSys.getextensionname(objFolder & "\" & objFile.name)) = ucase(strExtension) then
   ''Get Include
   For mysubLoop = 0 to ubound(arrFilterType)  ' Manage Includes
    If ucase(arrFilterType(mysubLoop)) = "INCLUDE" then
     If InStr(1, ucase(objFolder) & "\" & ucase(objFile.name) ,ucase( arrFilterString(mysubLoop)),vbTextCompare) Then 'Check if type is include
    	'wscript.echo "INCLUDE " & objFile.name      	
      boolMark=true
     End IF	
    End If	
   Next
   For mysubLoop = 0 to ubound(arrFilterType)  ' Manage Excludes
    If ucase(arrFilterType(mysubLoop)) = "EXCLUDE" then
     If InStr(1, ucase(objFolder) & "\" & ucase(objFile.name) ,ucase(arrFilterString(mysubLoop)),vbTextCompare) Then 'Check if type is exclude
    	'wscript.echo "EXCLUDE " & objFile.name      	
    	boolMark=false
     End IF	
    End If	
   Next
   If (boolMark) then 
    iSourceFilesCnt=iSourceFilesCnt+1
    redim preserve aSourceFiles(iSourceFilesCnt)
    redim preserve aSourceSize(iSourceFilesCnt)
    aSourceFiles(iSourceFilesCnt)=objFolder & "\" & objFile.name
    aSourceSize(iSourceFilesCnt)=int(objFile.size/1024/1024) 'Use MB as internal unit
   End If 
  End If 
 Next
 For Each objSubFolder in objFolder.subFolders 'Recusive function call
  Call subGetSourceFiles(strFolder & "\" & objSubFolder.name)
 Next
 Set objFileSys = Nothing
End Sub

function funcGetFreeSpace(strPath)
 'Get free disk space using drive letter or UNC paths - 1.1
 Dim objFSO,objDisk,intFreeSpace
 Set objFSo  = CreateObject("Scripting.FileSystemObject") 
 Set objDisk = objFSo.GetDrive(objFSo.GetDriveName(strPath))
 intFreeSpace=int(objDisk.FreeSpace/1024/1024) 'Return MB
 funcGetFreeSpace=intFreeSpace
 Set objFSo = Nothing
 Set objDisk = Nothing
End Function


Function funcGetFileBaseName(strPath)
 funcGetFileBaseName = Mid(strPath, InStrRev(strPath, "\") + 1)
End Function

Function funcGetHostName
 dim wshNetwork
 Set wshNetwork = CreateObject("WScript.Network") 
 funcGetHostName = wshNetwork.ComputerName
 Set wshNetwork = Nothing
End Function


sub subCopyVHD
 subCreateDirs(strTargetPath & funcGetHostName) 'Add subdir of our Hostname
 dim fso
 Set fso = CreateObject("Scripting.FileSystemObject")
 For myLoop = 1 To UBound(aSourceFiles)
  syslog "[COPY] " & funcAddTrailingBackslash(strConfVSSPath)  & funcSkipDriveLetter(aSourceFiles(myLoop))
  syslog "       To:" & strTargetPath & funcGetHostName & "\" & getDate & "-" & funcGetFileBaseName(aSourceFiles(myLoop)) & " (" & int(aSourceSize(myLoop)/1024) & "GB)"
	if (int(funcGetFreeSpace(strTargetPath)/1024) - int(aSourceSize(myLoop)/1024)) > intMinFreeDiskSpaceGB  then
   'syslog "[INF] disk space left (Available:" & int(funcGetFreeSpace(strTargetPath)/1024) & " / Required: " & int(aSourceSize(myLoop)/1024) & " / MinFree: " & intMinFreeDiskSpaceGB & ")"
   fso.copyfile funcAddTrailingBackslash(strConfVSSPath)  & funcSkipDriveLetter(aSourceFiles(myLoop)), strTargetPath & funcGetHostName & "\" & getDate & "-" &  funcGetFileBaseName(aSourceFiles(myLoop)) , True
   syslog funcRetentionData(strTargetPath & funcGetHostName,funcGetFileName(aSourceFiles(myLoop)),intDataRetention)
  Else 
   syslog "[FAIL] No more disk space left (Available:" & int(funcGetFreeSpace(strTargetPath)/1024) & " / Required: " & int(aSourceSize(myLoop)/1024) & " / MinFree: " & intMinFreeDiskSpaceGB & ")"
   boolErrorOccured=true
  End if
 Next   
End Sub 


Sub subCreateDirs( MyDirName )
 Dim arrDirs, i, idxFirst, objFSO, strDir, strDirBuild
 Set objFSO = CreateObject( "Scripting.FileSystemObject" )
 strDir = objFSO.GetAbsolutePathName( MyDirName )
 arrDirs = Split( strDir, "\" )
 If Left( strDir, 2 ) = "\\" Then
  strDirBuild = "\\" & arrDirs(2) & "\" & arrDirs(3) & "\"
  idxFirst    = 4
 Else
  strDirBuild = arrDirs(0) & "\"
  idxFirst    = 1
 End If

 For i = idxFirst to Ubound( arrDirs )
  strDirBuild = objFSO.BuildPath( strDirBuild, arrDirs(i) )
  If Not objFSO.FolderExists( strDirBuild ) Then
   objFSO.CreateFolder strDirBuild
  End if
 Next
 Set objFSO= Nothing
End Sub



Function funcRetentionData(strPath,strPattern,intAmount)
 'Delete files by pattern and amount of retention data
 'strPath =       Path to file(s)
 'strPattern =    Pattern to search
 'intAmount  =  Amount of data to keepOptional search for extension too, leave empty to skip
 Dim getFileFso,objList,f,strCurDate,intCnt
 'wscript.echo strPath
 If (intAmount < 1) then
  syslog"[WARN] Ingoring retention of 1 data file"
  exit function
 End if
 Set getFileFso = CreateObject("Scripting.FileSystemObject")
 Set objList = CreateObject("ADOR.Recordset")
 objList.Fields.Append "name", 200, 255
 objList.Fields.Append "date", 7
 objList.Open
 For Each f In getFileFso.GetFolder(strPath).Files
  objList.AddNew
  'wscript.echo "add" & strPath & f.Name & " - " &  f.DateLastModified
  objList("name").Value = f.Name
  objList("date").Value = f.DateLastModified
  objList.Update
 Next
 objList.Sort = "date DESC"
 objList.MoveFirst
 Do Until objList.EOF
  If InStr(1,ucase(objList("name")), ucase(strPattern),vbTextCompare) > 0 Then  'Match file pattern
   intCnt=intCnt+1
   if intCnt > intAmount then
    syslog "[INF] Automatic retention: Delete data: " & strPath & "\" & objList("name") ''& " -Pattern " & strPattern
    getFileFso.deletefile strPath & "\" & objList("name")
   else 
    syslog "[INF] Automatic retention: Keed data: " & strPath & "\" & objList("name") ''& " -Pattern " & strPattern
   end if
  End If	
  objList.MoveNext
 Loop
 objList.Close
End Function


function funcGetFileName(strName)
 'Get fthe filename out of the path, sample: c:\test\vm-test.vhdx would return vm-test.vhdx
 dim objFSO
 Set objFSO = CreateObject("Scripting.FileSystemObject")
 funcGetFileName= objFSO.GetFileName(strName)
 Set objFSO = Nothing
end function


sub subDeleteVSS(strDeviceObject)
  Dim objWMI, snapshots, snapshot, sDate, vDate, deleteSnapshot
  Set objWMI = GetObject("winmgmts://localhost/root/cimv2")
  Set snapshots = objWMI.ExecQuery("select * from Win32_ShadowCopy")
  for each snapshot in snapshots
   If snapshot.DeviceObject = strDeviceObject Then
    syslog"----- Deleting VSS: " & strDeviceObject
    ''Delete: snapshot.Delete_()
    snapshot.Delete_()
    exit for
   End IF
  Next
  Set objWMI = Nothing
  Set snapshots = Nothing
End sub

Function getLogDate '1.2
 'Return the the date/time in the format of yyyy-mm-dd hh:mm:ss including leading zeros
 dim strDate
 strDate = DatePart("yyyy",Date,4) & "-"  _
        & Right("0" & DatePart("m",Date), 2) & "-"  _
        & Right("0" & DatePart("d",Date), 2) & " " _
        & Right("0" & DatePart("h",Time), 2) & ":" _
        & Right("0" & DatePart("m",Time), 2) & ":" _
        & Right("0" & DatePart("s",Time), 2) 
 getLogDate = strDate        
End Function
