#!/usr/bin/perl
#Fail2Nft - The script to block unwanted ssh/smtp/imap/ftp login attemps
#See also: https://coolgeo.org/fail2nft

# This program is distributed in the hope that it will be useful,
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
# OTHER DEALINGS IN THE SOFTWARE.    

#INFO: Notepad++ was used to write this script

#Version history
#1.0 - going live							- 01Nov2019
#1.1 - Minor fix in CheckIP 				- 04May2021
#1.2 - Add SMTP attacks (Module mail.pm)  	- 02Sep2021
#1.3 - Add Multiple IP to Country Provider 	- 01Oct2021
#1.4 - Add Grafana logs as log source       - 04Jan2023
#      only v4 has been tested so far
#1.5 - Add splunk - Aug-Oct2023

#Load required modules! 
use strict;
use Module::Load::Conditional qw[can_load]; #Used to check if the syslog module has been installed
use lib "/usr/local/fail2nft/modules/"; #Embedding our own modules
use DBI;                                #Used to read SQLite data
use POSIX;                              #POSIX functions such as strftime
use LWP::UserAgent;                     #Used to fetch ip/geo data
use HTTP::Request::Common qw(GET);      #Used to convert log dates to epoch times
use XML::Simple;                        #Used to read the configuration file
use Data::Dumper;                       #Used to print hash data to debug, optional
use Getopt::Long;                       #Used to read command params
use JSON qw( );                         #Used to read our live nft environment / dumps
use File::Path qw(make_path);           #Check/Create Logfile Dir
use Net::IP ':PROC';                    #Used to determine the type of IP
use HTTP::Date qw/str2time/;            #Used to convert log date to epoch date
use Proc::ProcessTable;                 #Used for process management
use File::Basename;                     #Used for process management
use auth;                               #Use auth.pm which has been defined in "use lib"
use vsftpd;                             #Use vsftpd.pm which has been defined in "use lib"
use mail;                               #Use mail.pm which has been defined in "use lib"
use grafana;                            #Use grafana.pm which has been defined in "use lib" -> 1.4
use splunk;                             #Use splunk.pm which has been defined in "use lib" -> 1.5
use Mail::Sendmail;                     #Used to send simple mails  
use Sys::Hostname;                      #Used to get my hostname

#Check for missing Module Net::Syslog
my ($usesyslog); 
my $use_list = { 'Net::Syslog'};        #Prepared for scalar usage
if (can_load( modules => $use_list )) {  $usesyslog=1; } #disable syslog (Net::Syslog) in case if it this mod is not installed


#############
# Init  Vars
#############
my (@whitelistip,$nftStr,@syslogip,$strsyslog,%hIPBlock,%ReturnHash,%hIPData);
my (%AUTH_ReturnHash,%FTP_ReturnHash,%MAIL_ReturnHash);
my ($Table_IPV4,$Set_IPV4_drop,$Set_IPV4_accept,$Discard_Private_IPAddress);
my ($Table_IPV6,$Set_IPV6_drop,$Set_IPV6_accept,$procPID,$procSTIME);
my ($On_Success_Timer,$On_Fail_Timer,$On_Fail_Timer_Init,$Login_Fail_Counter,$Max_Reverse_Time,$On_Fail_Double_Timer,$On_Success_Renew);
my ($Delete_Inactive_Records,$Set_vsftp,$Process_Timeout,$Reset_Record_Counter,$GeoIP_Connect_Failure_Max,$tmpMsg); 
my ($flag_test,$flag_help,$flag_sqlite,$flag_list,$flag_json,$flag_init,$flag_drop,$flag_accept,$flag_check,$opt_D,$flag_reset,$flag_version,$flag_verbose,$flag_testmail);
my ($LoggingEnable,$Loggingpath,$GeoIP_URL,$GeoIP_NAME,$GeoIP_KEY,@syslogip,$cntd,$tmpNFTNameTable,$tmpNFTTarget,$tmpIPVer,$tmpIPType,$cnt_drop,$Set_mail,$flag_add,$opt_IP,$opt_time);
my ($EventMailTo,$EventMailFrom,$EventMailSMTP,$EventMailUser,$EventMailPassword,$EventMailLevel);
my (%CountryCode_Enable_DoubleTimer,%CountryCode_Set_Timer,%ASNName_Enable_DoubleTimer,%ASNName_Set_Timer);
#1.4
my ($Set_grafana,$cnt_drop_grafana,%Grafana_ReturnHash,$cnt_drop_grafana);
#1.5
my ($Set_splunk,$cnt_drop_splunk,%Splunk_ReturnHash,$cnt_drop_splunk); #1.5

########################################
#       INITIALIZATION TASKS
########################################
my $version = "1.5";
my $driver   = "SQLite";
my $appname  = "fail2nft";  #suffix to be used for log and conf naming
my $cnt_accept=0;  #Set def counter
my $cnt_drop=0;    #Set def counter 
my $Tmp_GeoIP_Connect_Failure_Max=3;  #Default var, will get set in case of an older config schema
my $configuration="/usr/local/fail2nft/$appname.xml";
my $databasepath="/usr/local/fail2nft/$appname.db";
&ReadConfig($configuration); # Getting xml configuration ($Conffile)
#--- Create logdir if it does not exist
if ($LoggingEnable == 1){
 if (!-e $Loggingpath) {
 	make_path $Loggingpath, {owner=>'nobody', group=>'nogroup'};
 }        
} 
#--- Cleaning tmp directory on startup
if (-e "/tmp/$appname"."error.log") {
	unlink "/tmp/$appname"."error.log";
}	
my $TimeStartup=time;
my $hostname = hostname;
my $myProcess=basename($0);

if (!-e $databasepath) {   #1.3
 $tmpMsg="Info: Database does not exist, we will try to create one: $databasepath";
}

my $dbd = "DBI:$driver:dbname=$databasepath";                               #***  Global DB handler *** 
my $dbh = DBI->connect($dbd, ,, { RaiseError => 1 }) or die $DBI::errstr; #Global DB connection, we will leave this open for the entire routine
GetOptions
	(
	 "h!" => \$flag_help, 
	 "ch!" => \$flag_check, 
	 "T!" => \$flag_test, 
	 "S!" => \$flag_sqlite, 
	 "L!" => \$flag_list, 
	 "a!" => \$flag_accept, 
	 "d!" => \$flag_drop, 
	 "json!" => \$flag_json, 
	 "I!" => \$flag_init, 
	 "delete=s" => \$opt_D, 
	 "r!" => \$flag_reset, 
	 "version!" => \$flag_version, 
	 "v!" => \$flag_verbose, 
	 "add!" => \$flag_add, 
	 "ip=s" => \$opt_IP, 
	 "time=i" => \$opt_time, 
	 "testmail!" => \$flag_testmail, 
);
#############################################
#######  FIN INITIALIZATION TASKS  ##########
#############################################



#################################
#######  CHECK PARAMS  ##########
#################################


#------Want at least one parameter given
if ((!$flag_check) and (!$flag_help) and (!$flag_list) and (!$flag_test) and (!$opt_D) and (!$flag_reset) and (!$flag_init) and (!$flag_version)  and (!$flag_add)and (!$flag_testmail))  {
	&print_help;
	exit;
}

if ($flag_version) {
	print "$appname version $version\n";
	exit;
}	

######## Select options ##########
&subTestSQLite;
if ($flag_help) {
	&print_help;
	exit;
}


if ($flag_sqlite) {
	&subTestSQLite;
	exit;
}


if ($flag_list) {
	&subListData;
	exit;
}

if ($flag_init) {
	&subInitNFT;
	exit;
}


if ($opt_D) {
 &subDeleteIP($opt_D);
 exit;
} 


if ($flag_reset) {
 &subReset;
 exit;
}

if ($flag_testmail) {
 &TestEmail;
 exit;
}


if ($flag_add) {
 if ((!$flag_accept) and (!$flag_drop)) {
	print "\nPlease specify the rule target with -a (accept) or -d (drop)\n\n";
	&print_help;
	exit;
 } else {
  if (!$opt_IP) {	
 	 print "\nPlease specify the target IP Address with -ip\n\n";
 	 &print_help;
 	 exit;
  } else {
   if (!$opt_time) {
    print "\nPlease specify the timer (in hours) for how long the rule should be active\n\n";
    &print_help;
    exit;
   } else {
    my $tmpTarget;
    if ($flag_accept) {$tmpTarget="a";}
    if ($flag_drop) {$tmpTarget="d";}
    &subAddIP($tmpTarget,$opt_IP,$opt_time); 
    exit;  		
   }  #if (!$opt_time) 
  }   #if (!$opt_IP) 
 }    #if ((!$flag_accept) and (!$flag_drop)) 
}     #if ($flag_add)


if ($flag_test) {
  if ($flag_json) {
	 print "\n*** NFT Dump ***\n";
	 print subTestConfiguration($Table_IPV4,$Set_IPV4_drop,4,1,1) ."\n";
	 print "\n*** Finished NFT Dump ***\n";
	 exit
  }
  if (($Table_IPV4) and ($Set_IPV4_drop)) {
   print "\nTesting v4 drop rules:\n";
   print subTestConfiguration($Table_IPV4,$Set_IPV4_drop,4,1,0) ."\n";
  } 
  if (($Table_IPV4) and ($Set_IPV4_accept)) {
	 print "\nTesting v4 accept rules: \n";
	 print subTestConfiguration($Table_IPV4,$Set_IPV4_accept,4,1,0) ."\n";
  } 
  if (($Table_IPV6) and ($Set_IPV6_drop)) {
	print "\nTesting v6 drop rules: \n";
	#exit;
	print subTestConfiguration($Table_IPV6,$Set_IPV6_drop,6,1,0) ."\n";
  } 
  if (($Table_IPV6) and ($Set_IPV6_accept)) {
   print "\nTesting v6 accept rules: \n";
   #exit;
   print subTestConfiguration($Table_IPV6,$Set_IPV6_accept,6,1,0) ."\n";
  } 
  exit;
}  #if ($flag_test)
##########################  FIN PARAMS  ##################################


#########################
#### Process Control ####
#########################
syslog(" ");
syslog("Starting Fail2Nft Version $version ");


if ($tmpMsg) {   #1.3
 syslog("Information: $tmpMsg");
}

($procPID,$procSTIME)=GetProcessInfo($myProcess,$$);  #PROCESS CONTROL - Make sure that our process is the only one, therefore check the process table 
if ($procPID) {
 if ( $procSTIME>$Process_Timeout ) { 
  syslog ("KILL $procPID as the timeout has been reached");
  kill 9, $procPID;                    # must it be 9 (SIGKILL)? 
  my $gone_pid = waitpid $procPID, 0;
 } else{
 	syslog("Another process of $myProcess is already running since $procSTIME seconds, will wait for " . ($Process_Timeout-$procSTIME) . " seconds. NOW ABORT and come back later");
 	exit;
 }
} else {
	syslog("Proc control: OK");
}

################
#### BEGIN  ####
################
&subCheckConfigVersion;  #Config version compatibility check - 1.3
&subTestSQLite;          #Check Tables and create them if needed
&subCheckExpiredRecords; #Reset expired records, recommend to do at startup
&subInitStatus;          #Check/Set Status Table  
my ($TimeLastExec,$GeoConnectionFailure,$ClockTotal)=&subGetStatus; #Get the previous run time date so we can quickly hook within the log
if (subCheckForDayChange($TimeLastExec)) { #Run maintenance if a daychange occured (midnight)
 syslog("DAYCHANGE: Run Maintenance");
 &subDailyMaintenance;
}

if (subCheckForMonthChange($TimeLastExec)) { #Run maintenance if a month change occured (midnight)
 syslog("MONTHCHANGE: Run Maintenance");
  &subMonthlyMaintenance;
}

#################################################################################
if ($Set_mail) {
 my $cnt_drop_mail=0;	
 mail::ReadMailRecords($TimeLastExec,\%MAIL_ReturnHash);	#Get mail log  data (syslog)
 for my $IP (sort keys %{%MAIL_ReturnHash{'IP_DENY'}}) {
  if ($IP ne "-") {  #Need a real IP
   $cnt_drop++;
   $cnt_drop_mail++;
   $ReturnHash{'IP_DENY'}{$IP}.=$MAIL_ReturnHash{'IP_DENY'}{$IP};     #Merge data into global hash
   $ReturnHash{'LOG'}{$IP}='MAIL';                                    #Merge data into global hash
  } 
 }
 syslog ("-Read mail.log records ($cnt_drop_mail)");
 my %MAIL_ReturnHash; #release memory
}



#################################################################################
if ($Set_vsftp) {
 my $cnt_drop_ftp=0;
 vsftpd::ReadFTPRecords($TimeLastExec,\%FTP_ReturnHash);	#Get vsftp log  data (syslog)
 $ReturnHash{'DEBUG'}.=$FTP_ReturnHash{'DEBUG'};
 for my $IP (sort keys %{%FTP_ReturnHash{'IP_DENY'}}) {
  if ($IP ne "-") {  #Need a real IP
   $cnt_drop++;
   $cnt_drop_ftp++;
   $ReturnHash{'IP_DENY'}{$IP}.=$FTP_ReturnHash{'IP_DENY'}{$IP};    #Merge data into global hash
   $ReturnHash{'LOG'}{$IP}='FTP';                                   #Merge data into global hash
  } 
 }	
 syslog ("-Read vsftp.log records ($cnt_drop_ftp)");
 my %FTP_ReturnHash; #release memory
}
#################################################################################


#################################################################################
if ($Set_grafana) {   #1.4
 my $cnt_drop_grafana=0;
 grafana::ReadGrafanaRecords($TimeLastExec,\%Grafana_ReturnHash);	#Get grafana log data
 $ReturnHash{'DEBUG'}.=$Grafana_ReturnHash{'DEBUG'};
 #syslog("Log"  );
 syslog($ReturnHash{'DEBUG'} );
 for my $IP (sort keys %{%Grafana_ReturnHash{'IP_DENY'}}) {
  if ($IP ne "-") {  #Need a real IP
   $ReturnHash{'IP_DENY'}{$IP}.=$Grafana_ReturnHash{'IP_DENY'}{$IP};  #Merge data into global hash
   $ReturnHash{'LOG'}{$IP}='GRFA';                                    #Merge data into global hash
   $cnt_drop++;
   $cnt_drop_grafana++;
  } 
 }	
 syslog ("-Read grafana.log records ($cnt_drop_grafana)");
 my %Grafana_ReturnHash; #release memory
}


#################################################################################
if ($Set_splunk) {   #1.5
 my $cnt_drop_splunk=0;
 splunk::ReadSplunkRecords($TimeLastExec,\%Splunk_ReturnHash);	#Get grafana log data
 $ReturnHash{'DEBUG'}.=$Splunk_ReturnHash{'DEBUG'};
 #syslog("Log"  );
 syslog($ReturnHash{'DEBUG'} );
 for my $IP (sort keys %{%Splunk_ReturnHash{'IP_DENY'}}) {
  if ($IP ne "-") {  #Need a real IP
   $ReturnHash{'IP_DENY'}{$IP}.=$Splunk_ReturnHash{'IP_DENY'}{$IP};  #Merge data into global hash
   $ReturnHash{'LOG'}{$IP}='SPLU';                                    #Merge data into global hash
   $cnt_drop++;
   $cnt_drop_splunk++;
  } 
 }	
 syslog ("-Read splunk.log records ($cnt_drop_splunk)");
 my %Splunk_ReturnHash; #release memory
}


#################################################################################
my $cnt_drop_auth=0;
my $cnt_allow_auth=0;
my $cnt_allow_auth=0;
auth::ReadSyslogRecords($TimeLastExec,\%AUTH_ReturnHash);	#Get auth log data (syslog)
$ReturnHash{'DEBUG'}.=$AUTH_ReturnHash{'DEBUG'};
for my $IP (sort keys %{%AUTH_ReturnHash{'IP_DENY'}}) {
 if ($IP ne "-") {  #Need a real IP
  $ReturnHash{'IP_DENY'}{$IP}.=$AUTH_ReturnHash{'IP_DENY'}{$IP};  #Merge data into global hash
  $ReturnHash{'LOG'}{$IP}='AUTH';                                 #Merge data into global hash
  $cnt_drop++;
  $cnt_drop_auth++;
 } 
}	
for my $IP (sort keys %{%AUTH_ReturnHash{'IP_ALLOW'}}) {
 if ($IP ne "-") {  #Need a real IP
	$ReturnHash{'IP_ALLOW'}{$IP}.=$AUTH_ReturnHash{'IP_ALLOW'}{$IP}; #Merge data into global hash
	$ReturnHash{'LOG'}{$IP}='AUTH';                                  #Merge data into global hash
	$cnt_accept++;
	$cnt_allow_auth++;
 } 
}	
syslog ("-Read auth.log records ($cnt_drop_auth/$cnt_allow_auth)");
my %AUTH_ReturnHash; #release memory
#################################################################################

syslog ("=>Records Success: $cnt_accept");
syslog ("=>Records Fail: $cnt_drop");



##########################################################################################################
#successful authentications. They take precedence so we add them first to our control database and to nft
#########################################################################################################

for my $IP (sort keys %{$ReturnHash{'IP_ALLOW'}}) {
 $On_Fail_Timer=$On_Fail_Timer_Init; #Read previous value back to original
 if ($IP ne "-") {  #Need a real IP
  my $loginfo;
  if ($ReturnHash{'LOG'}{$IP}) {
  	$loginfo=$ReturnHash{'LOG'}{$IP};
  }
  ($tmpIPVer,$tmpIPType) = subGetIPInfo($IP); #Get IP Verison and Type
  if ($tmpIPVer == 4) { #Map IP Version to NFTable
   if ((!$Table_IPV4) or (!$Set_IPV4_drop) or (!$Set_IPV4_accept) ){
	 	syslog("WARNING: IP4 is not configured - abort");
	 	last;
	 }
	 $tmpNFTNameTable=$Table_IPV4;
	 $tmpNFTTarget=$Set_IPV4_accept;
  }	
  if ($tmpIPVer == 6) { #Map IP Version to NFTable
   if ((!$Table_IPV6) or (!$Set_IPV6_drop) or (!$Set_IPV6_accept) ){
	syslog("WARNING: IP6 is not configured - abort");
	last;
   }
   $tmpNFTNameTable=$Table_IPV6;
   $tmpNFTTarget=$Set_IPV6_accept;
  }	
  syslog("Allow: $IP - Cnt: $ReturnHash{'IP_ALLOW'}{$IP} -- Type $tmpIPVer - Log: $ReturnHash{'LOG'}{$IP}");
  if (($Discard_Private_IPAddress == 1) and (($tmpIPType eq 'PUBLIC') or ($tmpIPType eq 'GLOBAL-UNICAST')) or ($Discard_Private_IPAddress == 0)) {  #If set discard private ip address AND ip address is public
   my ($TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES,$PREVIOUS_LOCK_TIMER)=subCheckIP($IP);  #Check IP record
   syslog("Record Dump (INFO): $TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES ---> $PREVIOUS_LOCK_TIMER");
   if (!$TIME_LAST_SEEN) { #Record does NOT EXIST yet
    my ($detailIP_Country,$detailIP_ASN)=&getIPDetails($IP,$GeoConnectionFailure,$GeoIP_Connect_Failure_Max);  #Get Country & ASN
    syslog("Add data $detailIP_Country -- $detailIP_ASN -- Type: $tmpIPVer");
    my $cmdstat=subApplyNFT($IP,$tmpNFTNameTable,$tmpNFTTarget,$On_Success_Timer,$detailIP_Country,$detailIP_ASN,$tmpIPVer,$loginfo); #Apply NFT Rule
    &MessageViaEmail(1,$IP,$On_Success_Timer,$detailIP_Country,$detailIP_ASN,$tmpIPVer,$loginfo);
    &MessageViaSyslog("M=F2N LOCK=0 IP=$IP TIMER=$On_Success_Timer COUNTRY=$COUNTRY ASN=$ASN LOG=$loginfo");
    if (!$cmdstat) {
     syslog("NFT applied OK 1");
     &addtblIP($IP,$detailIP_Country,$detailIP_ASN,1,$On_Success_Timer,1,'0','1',$ReturnHash{'LOG'}{$IP});  #Add IPv4 Records to SQLite
    } else {
     syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
    }
   } else {  #if (!$TIME_LAST_SEEN)  #Record does NOT EXIST yet
    if ($APPLIED_TO_NFT == 0 ) {
     syslog("Record is not  set to NFT");
     my $cmdstat=subApplyNFT($IP,$tmpNFTNameTable,$tmpNFTTarget,$On_Success_Timer,$COUNTRY,$ASN,$tmpIPVer,$loginfo); #Apply NFT Rule and reuse COUNTRY/ASN information
     &MessageViaEmail(1,$IP,$On_Success_Timer,$COUNTRY,$ASN,$tmpIPVer,$loginfo);
     &MessageViaSyslog("M=F2N LOCK=0 IP=$IP TIMER=$On_Success_Timer COUNTRY=$COUNTRY ASN=$ASN LOG=$loginfo");
     if (!$cmdstat) {
      syslog("NFT applied OK");
      &subUpdateIP($IP,$COUNTRY,$ASN,$COUNTER_TOT,$On_Success_Timer,1,$COUNTER_NFT);  #Add IPv4 Records to SQLite
     } else {
      syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
     }      #if (!$cmdstat)
    }  else {
     syslog("IP already applied to NFT");
  	 if ($On_Success_Renew == 1) {
  	  my ($TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES,$PREVIOUS_LOCK_TIMER)=subCheckIP($IP);  #Check IP record
  	  &UpdateIPAllow($IP,$COUNTRY,$ASN,$loginfo);
  	 } 
    }	    #if ($APPLIED_TO_NFT == 0 )
   }        #if (!$TIME_LAST_SEEN)  #Record does NOT EXIST yet
  } else {  #if (($Discard_Private_IPAddress == 1) and ($tmpIPType eq 'PUBLIC') or ($Discard_Private_IPAddress == 0))   #If set discard private ip address AND ip address is private
   syslog("DISCARD PRIVATE IP: $IP");
  }         #if (($Discard_Private_IPAddress == 1) and ($tmpIPType eq 'PUBLIC') or ($Discard_Private_IPAddress == 0))   #If set discard private ip address AND ip address is private 
 }  #if ($IP ne "-") { 
}	        #for my $IP (sort keys %{$ReturnHash{'IP_ALLOW'}})



#######################################################################
#Failed authentications. We add them to our control database and to nft
#######################################################################
 
$tmpNFTNameTable=undef; #Reset previous stats to be secure
$tmpNFTTarget=undef;    #Reset previous stats to be secure

for my $IP (sort keys %{$ReturnHash{'IP_DENY'}}) {
 $On_Fail_Timer=$On_Fail_Timer_Init; #Read previous value back to original
 if ($IP ne "-") {  #Need a real IP
  my $loginfo;
  if ($ReturnHash{'LOG'}{$IP}) {
   $loginfo=$ReturnHash{'LOG'}{$IP};
  }
  if (!$ReturnHash{'IP_ALLOW'}{$IP}) { #Cross check our previous allowed IPs
   if (!subCheckWhitelist($IP) ) {
    $cntd++;
    ############################
    #          DENY IP
    ############################
    ($tmpIPVer,$tmpIPType) = subGetIPInfo($IP); #Get IP Verison and Type
    syslog("-IP Address Type: $tmpIPVer,$tmpIPType - $IP");
    if ($tmpIPVer == 4) { #Map IP Version to NFTable
   	 if ((!$Table_IPV4) or (!$Set_IPV4_drop) or (!$Set_IPV4_accept) ){
	  	syslog("WARNING: IP4 is not configured - abort");
	 	  last;
	   }
     $tmpNFTNameTable=$Table_IPV4;
     $tmpNFTTarget=$Set_IPV4_drop;
    }	
    if ($tmpIPVer == 6) {  #Map IP Version to NFTable
	 if ((!$Table_IPV6) or (!$Set_IPV6_drop) or (!$Set_IPV6_accept) ){
  	  syslog("WARNING: IP6 is not configured - abort");
	  last;
	 }
     $tmpNFTNameTable=$Table_IPV6;
     $tmpNFTTarget=$Set_IPV6_drop;
    }	
    syslog("-Record: $cntd/$cnt_drop");
    syslog("-Deny: $IP - Cnt: $ReturnHash{'IP_DENY'}{$IP} - Version: $tmpIPVer - Type: $tmpIPType");
    syslog("-Deny NFT Settings: $tmpNFTTarget / $tmpNFTNameTable");
    if (($Discard_Private_IPAddress == 1) and (($tmpIPType eq 'PUBLIC') or ($tmpIPType eq 'GLOBAL-UNICAST')) or ($Discard_Private_IPAddress == 0)) {  #If set discard private ip address AND ip address is public
     #$Login_Fail_Counter
     my ($TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES,$PREVIOUS_LOCK_TIMER)=subCheckIP($IP);  #Check IP record
     #syslog("Record Dump: $TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES == > $PREVIOUS_LOCK_TIMER");
     if (!$TIME_LAST_SEEN) { #Record does NOT EXIST yet!!!
      my ($detailIP_Country,$detailIP_ASN)=&getIPDetails($IP,$GeoConnectionFailure,$GeoIP_Connect_Failure_Max);  #Get Country & ASN
      for my $Code (sort keys %CountryCode_Set_Timer) { #check on per country level
       if ($detailIP_Country eq $Code) {
        syslog ("Country is matching on a per country level WE SET THE VERY FIRST BASE LEVEL AS WE TOLD SO: $IP/$Code - but overwrite $On_Fail_Timer with $CountryCode_Set_Timer{$Code}");
        $On_Fail_Timer=$CountryCode_Set_Timer{$Code};
       }	    #if ($COUNTRY eq $Code) 
      }      #for my $Code (sort keys %CountryCode_Set_Timer)  #check on per country level
      for my $Name (sort keys %ASNName_Set_Timer) { #check on per ASN level
       if ($detailIP_ASN eq $Name) {
        syslog ("ASN is matching on a per ASN level WE SET THE VERY FIRST BASE LEVEL AS WE TOLD SO: $IP/$Name - but overwrite $On_Fail_Timer with $ASNName_Set_Timer{$Name}");
        $On_Fail_Timer=$ASNName_Set_Timer{$Name};
       }	   #if ($detailIP_ASN eq $Name) 
      }      #for my $Name (sort keys %ASNName_Set_Timer)  #check on per ASN level
      syslog("$IP $detailIP_Country,$detailIP_ASN");
      if ($ReturnHash{'IP_DENY'}{$IP} >=$Login_Fail_Counter ) {
       syslog("First time seen and lock immediatley $Login_Fail_Counter/$ReturnHash{'IP_DENY'}{$IP}");
       my $cmdstat=subApplyNFT($IP,$tmpNFTNameTable,$tmpNFTTarget,$On_Fail_Timer,$detailIP_Country,$detailIP_ASN,$tmpIPVer,$loginfo); #Apply NFT Rule and reuse COUNTRY/ASN information
       &MessageViaEmail(2,$IP,$On_Fail_Timer,$detailIP_Country,$detailIP_ASN,$tmpIPVer,$loginfo);
       &MessageViaSyslog("M=F2N LOCK=1 IP=$IP TIMER=$On_Fail_Timer COUNTRY=$detailIP_Country ASN=$detailIP_ASN LOG=$loginfo");
       if (!$cmdstat) {
        syslog("NFT applied OK 2 -- First Time Init: $On_Fail_Timer");
        &addtblIP($IP,$detailIP_Country,$detailIP_ASN,0,$On_Fail_Timer,$ReturnHash{'IP_DENY'}{$IP},$On_Fail_Timer,'1',$ReturnHash{'LOG'}{$IP});  #Add IP Records to SQLite
       } else {
        syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
       }
      } else { #if ($ReturnHash{'IP_DENY'}{$IP} >=$Login_Fail_Counter )
       #$ReturnHash{'IP_DENY'}{'123'}=1; #????
       syslog("Record Dump (Locked XX Seen) : $IP : $TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$detailIP_Country,$detailIP_ASN ---  $Login_Fail_Counter / $ReturnHash{'IP_DENY'}{$IP}");
       #&addtblIP($IP,$detailIP_Country,$detailIP_ASN,0,$On_Fail_Timer,$ReturnHash{'IP_DENY'}{$IP},'0','0',$ReturnHash{'LOG'}{$IP});  #Add IP Records to SQLite - no NFT apply yet as the max counter is not reached yet
       &addtblIP($IP,$detailIP_Country,$detailIP_ASN,0,0,$ReturnHash{'IP_DENY'}{$IP},'0','0',$ReturnHash{'LOG'}{$IP});  #Add IP Records to SQLite - no NFT apply yet as the max counter is not reached yet
      }         #if ($Login_Fail_Counter < $ReturnHash{'IP_DENY'}{$IP}) 
     } else {   #if (!$TIME_LAST_SEEN)  #Record does NOT EXIST yet
      if ($APPLIED_TO_NFT == 0 ) {   #NOT APPLIED YET
       my ($LevelOverwrite);
       ##***********    Overwrite per Country or ASN Level  	 ************** 
       for my $Code (sort keys %CountryCode_Set_Timer) { #check on per country level
        if ($COUNTRY eq $Code) {
         $LevelOverwrite=1;  #Skipt further modes
         if ($PREVIOUS_LOCK_TIMER == 0 ) {
          syslog ("Country is matching on a per country level WE SET THE BASE LEVEl AS THIS IS THE FIRST INIT: $IP/$Code - but overwrite $On_Fail_Timer with $CountryCode_Set_Timer{$Code}");
          $On_Fail_Timer=$CountryCode_Set_Timer{$Code};
         } else {      #if ($PREVIOUS_LOCK_TIMER == 0 ) {
          if ($CountryCode_Enable_DoubleTimer{$Code}) { #Double timer!
           syslog ("Country is matching on a per country level WE DOUBLE : $IP/$Code - overwrite $On_Fail_Timer with $CountryCode_Set_Timer{$Code} - Previous: $PREVIOUS_LOCK_TIMER NOW: " . ($PREVIOUS_LOCK_TIMER * 2));
           $On_Fail_Timer=($PREVIOUS_LOCK_TIMER * 2);	#Dynamic increase of lock time - double the value
          } else {
           syslog ("Country is matching on a per country level WE SET THE BASE LEVEL AS WE TOLD SO: $IP/$Code - but overwrite $On_Fail_Timer with $CountryCode_Set_Timer{$Code}");
           $On_Fail_Timer=$CountryCode_Set_Timer{$Code};
          }	  #if ($CountryCode_Enable_DoubleTimer{$Code})  #Double timer!
         }    #if ($PREVIOUS_LOCK_TIMER == 0 ) 	
        }	    #if ($COUNTRY eq $Code) 
       }      #for my $Code (sort keys %CountryCode_Set_Timer)  #check on per country level
       for my $Name (sort keys %ASNName_Set_Timer) { #check on per ASN level
        if ($ASN eq $Name) {
        	$LevelOverwrite=1;  #Skip further modes
         if ($PREVIOUS_LOCK_TIMER == 0 ) {
          syslog ("ASN is matching on a per ASN level WE SET THE BASE LEVEl AS THIS IS THE FIRST INIT: $IP/$Name - but overwrite $On_Fail_Timer with $ASNName_Set_Timer{$Name}");
          $On_Fail_Timer=$ASNName_Set_Timer{$Name};
         } else {      #if ($PREVIOUS_LOCK_TIMER == 0 ) {
          if ($ASNName_Enable_DoubleTimer{$Name}) { #Double timer!
           syslog ("ASN is matching on a per ASN level WE DOUBLE : $IP/$Name - overwrite $On_Fail_Timer with $ASNName_Set_Timer{$Name} - Previous: $PREVIOUS_LOCK_TIMER NOW: " . ($PREVIOUS_LOCK_TIMER * 2));
           $On_Fail_Timer=($PREVIOUS_LOCK_TIMER * 2);	#Dynamic increase of lock time - double the value
          } else {
           syslog ("ASN is matching on a per ASN level WE SET THE BASE LEVEL AS WE TOLD SO: $IP/$Name - but overwrite $On_Fail_Timer with $ASNName_Set_Timer{$Name}");
           $On_Fail_Timer=$ASNName_Set_Timer{$Name};
          }	  #if ($ASNName_Enable_DoubleTimer{$Name})  #Double timer!
         }    #if ($PREVIOUS_LOCK_TIMER == 0 ) 	
        }	    #if ($ASN eq $Name)
       }      #for my $Name (sort keys %ASNName_Set_Timer)  #check on per ASN level
     	 #***********    FIN Overwrite per Country or ASN Level  	 **************
       #Manage $PREVIOUS_LOCK_TIMER
       if ( ($On_Fail_Double_Timer == 1) and (!$LevelOverwrite)) {
        if ($PREVIOUS_LOCK_TIMER == 0 ) {
         $PREVIOUS_LOCK_TIMER = $On_Fail_Timer; 
         syslog("Set PREVIOUS_LOCK_TIMER to On_Fail_Timer - do we apply? should not come to here");
        } else {
         $On_Fail_Timer=($PREVIOUS_LOCK_TIMER * 2);	#Dynamic increase of lock time - double the value
        }
        syslog("DOUBLE Timer -- $IP");
       }    
       syslog("Record Dump (Locked Seen) : $TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES");
       if ( $COUNTER_CUR >= $Login_Fail_Counter) {	
       	syslog("Record is not  set to NFT");
       	syslog("Record Dump (Locked XXYY Seen) : $IP : $TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES ---  $Login_Fail_Counter / $ReturnHash{'IP_DENY'}{$IP}");
        my $cmdstat=subApplyNFT($IP,$tmpNFTNameTable,$tmpNFTTarget,$On_Fail_Timer,$COUNTRY,$ASN,$tmpIPVer,$loginfo); #Apply NFT Rule and reuse COUNTRY/ASN information
        &MessageViaEmail(2,$IP,$On_Fail_Timer,$COUNTRY,$ASN,$tmpIPVer,$loginfo);
        &MessageViaSyslog("M=F2N LOCK=1 IP=$IP TIMER=$On_Fail_Timer COUNTRY=$COUNTRY ASN=$ASN LOG=$loginfo");
        if (!$cmdstat) {
         syslog("NFT applied OK 3");
         &subUpdateIP($IP,$COUNTRY,$ASN,$COUNTER_TOT,$On_Fail_Timer,0,$COUNTER_NFT);  #Add IPv4 Records to SQLite
        } else {  #if (!$cmdstat)
         syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
        }          #if (!$cmdstat)
       }  else {   #if ($Login_Fail_Counter < $ReturnHash{'IP_DENY'}{$IP}) - WE UPDATE ONLY OUR DATABASE -NO ACTION YET
        syslog("New Record $IP $Login_Fail_Counter / $COUNTER_CUR / TOT " .($COUNTER_CUR + $ReturnHash{'IP_DENY'}{$IP}));
        &subIncrementCounterIP($IP,($COUNTER_CUR + $ReturnHash{'IP_DENY'}{$IP}),$COUNTER_TOT);
        if (($COUNTER_CUR + $ReturnHash{'IP_DENY'}{$IP}) >= $Login_Fail_Counter) {
         my $cmdstat=subApplyNFT($IP,$tmpNFTNameTable,$tmpNFTTarget,$On_Fail_Timer,$COUNTRY,$ASN,$tmpIPVer,$loginfo); #Apply NFT Rule and reuse COUNTRY/ASN information
         &MessageViaEmail(2,$IP,$On_Fail_Timer,$COUNTRY,$ASN,$tmpIPVer,$loginfo);
         &MessageViaSyslog("M=F2N LOCK=1 IP=$IP TIMER=$On_Fail_Timer COUNTRY=$COUNTRY ASN=$ASN LOG=$loginfo");
         if (!$cmdstat) {
          syslog("NFT applied OK 4");
          &subUpdateIP($IP,$COUNTRY,$ASN,$COUNTER_TOT,$On_Fail_Timer,0,$COUNTER_NFT);  #Add IPv4 Records to SQLite
         } else {  #if (!$cmdstat)
          syslog("NFT reports an error, see /tmp/$appname"."__error.log ");
         }       #if (!$cmdstat)
        }        #if (($COUNTER_CUR + $ReturnHash{'IP_DENY'}{$IP}) >= $Login_Fail_Counter) 
       }         ##if ($Login_Fail_Counter < $ReturnHash{'IP_DENY'}{$IP}) 
      }          #if ($APPLIED_TO_NFT == 0) /   #Record does NOT EXIST yet / new record appears here
     }	         #if (!$TIME_LAST_SEEN)  #Record does NOT EXIST yet
    } else {     #if (($Discard_Private_IPAddress == 1) and ($tmpIPType eq 'PRIVATE')) #If set discard private ip address AND ip address is private
     syslog("DISCARD PRIVATE IP: $IP");
    }            #if (($Discard_Private_IPAddress == 1) and ($tmpIPType eq 'PRIVATE')) #If set discard private ip address AND ip address is private
   } else {     #if (!subCheckWhitelist($IP) )
    syslog("SKIP IP: whitleist is set - $IP");
   }             #if (!subCheckWhitelist($IP) )
  } else {       #if (!$ReturnHash{'IP_ALLOW'}{$IP})  #Cross check our previous allowed IPs
   syslog("SKIP TO BLOCK IP because i received success during my interval check - $IP");
   if ($On_Success_Renew == 1) {
    my ($TIME_LAST_SEEN,$COUNTER_CUR,$COUNTER_TOT,$COUNTER_NFT,$APPLIED_TO_NFT,$COUNTRY,$ASN,$TIME_EXPIRES,$PREVIOUS_LOCK_TIMER)=subCheckIP($IP);  #Check IP record
    &UpdateIPAllow($IP,$COUNTRY,$ASN,$loginfo);
   } 
  }              #if (!$ReturnHash{'IP_ALLOW'}{$IP})  #Cross check our previous allowed IPs
 }               #if ($IP ne "-") {
}                #for my $IP (sort keys %{$ReturnHash{'IP_DENY'}}) 

&subSetStatus($TimeStartup,$ClockTotal);          #Set Status Table  

###################
###################
#       END
###################
###################

sub subGetStatus {
 #Params: None
 #Return
 #RUNTIME_LAST		    Last epoch time of execution - control var
 #COUNT_FAILURE_GEOIP	Failure count, control var
 #ClockTotal
 my $sqlstr = "SELECT RUNTIME_LAST,COUNT_FAILURE_GEOIP,Clock_Total from tblStatus";
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
 (my ($sqlRUNTIME_LAST,$sqlCOUNT_FAILURE_GEOIP,$sqlClockTotal)=$sql->fetchrow_array);
 #print  "==>> $sqlRUNTIME_LAST,$sqlRUNTIME_GEOIP,$sqlCOUNT_FAILURE_GEOIP \n";
 return $sqlRUNTIME_LAST,$sqlCOUNT_FAILURE_GEOIP,$sqlClockTotal;
}


sub subInitStatus {
 #Check if tblStatus is set (the first record), if not then insert a default record
 #Params: None
 #Return  None
 my ($sqlstr,$sql,$curTime);
 #Check if a record exist, create and return if none
 $sqlstr = "SELECT count(*)  from tblStatus";
 $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
 (my ($sqlCount)=$sql->fetchrow_array);
 if ($sqlCount == 0) {
   $curTime=(time-$Max_Reverse_Time);
   my $sqlstr = "Insert into tblStatus (RUNTIME_LAST,COUNT_FAILURE_GEOIP,CLOCK_TOTAL) values ($curTime,0,0);";
   my $sql = $dbh->prepare($sqlstr);
   $sql->execute; 
   return ($curTime,0,0)
 }
}


sub subSetStatus {
 #Set the actual time into tblStatus
 #Params: 
 #Startup time
 #Counter Total
 #Return  None
 #CLOCK_TOTAL
 my $sqlstr = "UPDATE tblStatus set RUNTIME_LAST=$_[0],CLOCK_TOTAL=" . ($_[1] + 1);
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
}

sub subCheckExpiredRecords {
 #Params: None
 #Return: None
 #Update all record if TIME_EXPIRES is older then now and apllied to NFT
 my $sqlstr = "Update tblIP set TIME_EXPIRES=0,APPLIED_TO_NFT=0,COUNTER_CUR=0 where APPLIED_TO_NFT = 1 and TIME_EXPIRES < " .time  . ";";
 syslog("Apply $sqlstr");
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; #Fire and forget ...
}

sub subApplyNFT {
 #$nftStr='nft add set ' .$Table_IPV4 . ' ' . $Set_IPV4_drop . ' \{type ipv4_addr \; flags timeout \; elements=\{'."$tmpIP".' timeout '.$tmpExpire.'s  comment \"' . "$COUNTRY/$ASN" . '\" \} \;\}';
 #Params
 #0: IP
 #1: Name of Table 
 #2: Name of Set (differ between Block/Allow)
 #3: Expiry
 #4: $COUNTRY
 #5: $ASN	
 #6: IP Version
 #$7 Log Info
 #Return: 
 # Def = error
 # Undef = OK
 
 my ($sub_nftStr,$errstr,$strComment,$tmpVer,$tmpType);
	
 if ($_[6] == 4) {	
	$tmpVer="ip";
	$tmpType='ipv4_addr';
 }
 
 if ($_[6] == 6) {	
	$tmpVer="ip6";
	$tmpType='ipv6_addr';
 }
 	
 if ($_[4]) {
 	$strComment="$_[4]";	
 }
 
 if ($_[5]) {
 	$strComment.="/$_[5]";	
 }

 if ($_[7]) {
 	$strComment.="/$_[7]";	
 }
 			
 #Todo: Check for internal IP's	
 #Todo: Check for whitelists	
 
 #SAMPLE: $nftStr='nft add set filter_v4 log2nft_drop  \{type ipv4_addr \; flags timeout \; elements=\{'."$tmpIP".' timeout '.'300'.'s  comment \"' . "$strComment" . '\" \} \;\}'; 
 if ($strComment) { #If Country OR ASN is given
  $sub_nftStr="/usr/sbin/nft add set $tmpVer " .$_[1] . ' ' . $_[2] . ' \{type '.$tmpType.' \; flags timeout \; elements=\{'."$_[0]".' timeout '.$_[3].'s  comment \"' . "$strComment" . '\" \} \;\}';
 } else {
 	  $sub_nftStr="/usr/sbin/nft add set $tmpVer " .$_[1] . ' ' . $_[2] . ' \{type '.$tmpType.' \; flags timeout \; elements=\{'."$_[0]".' timeout '.$_[3].'s  \} \;\}'; 
 }
 syslog("NFT: $sub_nftStr ");
 my $output = system("$sub_nftStr 2>>/tmp/$appname"."_error.log"); #Print the error message(s) to /tmp
 if ($output ne 0){
  	return 1;
 } 
 return undef;
}  # subApplyNFT


sub subUpdateIP {
 #Params
 #0: IP
 #1: $COUNTRY
 #2: $ASN	
 #3: COUNTER_TOT
 #4: TIME_EXPIRES
 #5: Drop (0) or Accept (1)
 #6: Counter NFT
 
 my ($cntNFT);
 if ($_[5] == 0 ) {
 	$cntNFT=$_[6]+1;
 } else {
  $cntNFT=$_[6];
 } 
 
 #Return: None
 my $sqlstr = "Update tblIP set TIME_LAST_SEEN='".time."', COUNTER_CUR=0, COUNTER_TOT=".($_[3]+1).", APPLIED_TO_NFT=1, TIME_EXPIRES=" . ($_[4] + time) . ",FLAG_WHITE=$_[5],COUNTER_NFT=$cntNFT,PREVIOUS_LOCK_TIMER=$_[4]  where IP = '$_[0]';";
 syslog("Apply: $sqlstr");
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; #Fire and forget ...
	
}  #subUpdateIP


sub subIncrementCounterIP {
 #Params
 #0: IP
 #1: Counter
 #2: Counter Tot
 
 my ($cntNFT);
 if ($_[5] == 0 ) {
 	$cntNFT=$_[6]+1;
 } else {
  $cntNFT=$_[6];
 } 
 
 #Return: None
 #my $sqlstr = "Update tblIP set TIME_LAST_SEEN='".time."', COUNTER_CUR=".($_[1]+1).", COUNTER_TOT=".($_[2]+1)."  where IP = '$_[0]';";
 my $sqlstr = "Update tblIP set TIME_LAST_SEEN='".time."', COUNTER_CUR=$_[1], COUNTER_TOT=".($_[2]+1)."  where IP = '$_[0]';";
 syslog("Apply: $sqlstr");
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; #Fire and forget ...
	
}  #subIncrementCounterIP

sub addtblIP {
 #Params
 #0: IP
 #1: $COUNTRY
 #2: $ASN	
 #3: FLAG_WHITE
 #4: TIME_EXPIRES
 #5: Current Counter
 #6: PREVIOUS_LOCK_TIMER
 #7: APPLIED_TO_NFT
 #8: LOG
 #Return: None
 #Note: If FLAG_WHITE is given then APPLIED_TO_NFT turns automatically to true
 #print "DEB 0 $_[0] \n";
 #print "DEB 1 $_[1] \n";
 #print "DEB 2 $_[2] \n";
 #print "DEB 3 $_[3] \n";
 #print "DEB 4 $_[4] \n";
 #print "DEB 5 $_[5] \n";
 #print "DEB 6 $_[6] \n";
 #print "DEB 7 $_[7] \n";
 #print "DEB 8 $_[8] \n";
 my $sqlstr = "INSERT INTO tblIP (IP,TIME_FIRST_SEEN,TIME_LAST_SEEN,TIME_EXPIRES,COUNTER_CUR,COUNTER_TOT,COUNTER_NFT,FLAG_WHITE,APPLIED_TO_NFT,COUNTRY,ASN,PREVIOUS_LOCK_TIMER,LOG)
           VALUES ('$_[0]', " .time . " ," . time . "," . ($_[4] + time) . ",$_[5],1,0,$_[3],$_[7],'$_[1]','$_[2]',$_[6],'$_[8]')";
 syslog("Apply to SQLITE: $sqlstr");
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; #Fire and forget ...

}  #addtblIP


sub subCheckIP {
  #Check the database for the given IPv4 address
	#Params
	#$_[0] = IPv4
	#Return if found
	#-Time last seen
	#-Counter_Cur
	#-Country
	#-ASN
  #-TIME_EXPIRES
  my $sqlstr = "select TIME_LAST_SEEN,COUNTER_CUR,COUNTER_TOT,COUNTER_NFT,APPLIED_TO_NFT,COUNTRY,ASN,TIME_EXPIRES,PREVIOUS_LOCK_TIMER from tblIP where IP = '$_[0]';";
  my $sql = $dbh->prepare($sqlstr);
  $sql->execute;
  (my ($sqlTIME_LAST_SEEN,$sqlCOUNTER_CUR,$sqlCOUNTER_TOT,$sqlCOUNTER_NFT,$sqlAPPLIED_TO_NFT,$sqlCOUNTRY,$sqlASN,$sqlTIME_EXPIRES,$sqlPREVIOUS_LOCK_TIMER)=$sql->fetchrow_array);
  return ($sqlTIME_LAST_SEEN,$sqlCOUNTER_CUR,$sqlCOUNTER_TOT,$sqlCOUNTER_NFT,$sqlAPPLIED_TO_NFT,$sqlCOUNTRY,$sqlASN,$sqlTIME_EXPIRES,$sqlPREVIOUS_LOCK_TIMER);
}  #sub subCheckIP


sub subCheckConfigVersion  {
 #Check/convert xml config schema lt 1.3
 if (!$GeoIP_NAME) {
  if ($GeoIP_URL =~ /xml.coolgeo.org/) { #Check xml config schema <1.3
	#syslog("-Old url is set in configuration (version less than 1.3), we will replace this to GeoIP_NAME=coolgeo.org")  ;
	$GeoIP_NAME="coolgeo.org";
  }
 }    #if (!$GeoIP_NAME) - var was not prsent before 1.3

 if (!$GeoIP_Connect_Failure_Max) {
  $GeoIP_Connect_Failure_Max=$Tmp_GeoIP_Connect_Failure_Max;
 }



}	  #sub subCheckConfigVersion 


sub subTestSQLite {
 # automaticaly create tables if they dont' exist
 
 #Table  =>tblIP<=  Field description :
 # IP 								 The IPv4/6 Address, length is 64bit to cover v6	
 # TIME_FIRST_SEEN		 Epoch Time of first seen
 # TIME_LAST_SEEN 		 Epoch Time of last seen
 # TIME_EXPIRES 			 Epoche time of expiration, used to reinit after reboot
 # PREVIOUS_LOCK_TIMER Remember the previous lock time
 # COUNTER_CUR INT		 Counter current - control var
 # COUNTER_TOT INT		 Counter total - stat var
 # COUNTER_NFT INT		 Counter nof nft applies - control var
 # FLAG_WHITE					 Identify IP Address to be whitelisted
 # APPLIED_TO_NFT		   Is currenctly applied - control var
 # COUNTRY 						 Country of IP address
 # ASN								 ASN of IP address
 # LOG 								 Name of the Logifile where the record has been found
 my $stmt = qq(CREATE TABLE IF NOT EXISTS tblIP
             (IP VARCHAR(64) PRIMARY KEY,
              TIME_FIRST_SEEN INT NOT NULL,
              TIME_LAST_SEEN INT NOT NULL,
              TIME_EXPIRES INT NOT NULL,
              PREVIOUS_LOCK_TIMER INT NOT NULL,
              COUNTER_CUR INT NOT NULL,
              COUNTER_TOT INT NOT NULL,
              COUNTER_NFT INT NOT NULL,
              FLAG_WHITE INT NOT NULL,
              APPLIED_TO_NFT INT NOT NULL,
              COUNTRY VARCHAR(2),
	            ASN VARCHAR(8), 	
              LOG VARCHAR(8) 	
             ););
 my $ret = $dbh->do($stmt);
 if($ret < 0) {
   syslog("Error Create Table IPV4 - $DBI::errstr");
 } else {
   #syslog("Create Table IPV4");
 }

 
 my $stmt = qq(CREATE INDEX IF NOT EXISTS expiry ON tblIP (
    APPLIED_TO_NFT,
    TIME_EXPIRES
   ););

 my $ret = $dbh->do($stmt);
 if($ret < 0) {
   #print STDERR $DBI::errstr;
   syslog("Error Create Index - $DBI::errstr");
 } else {
   #print STDERR "Table created successfully\n";
   #syslog("Create Table IPV4");
 }

 #Table  =>tblStatus<=  Field description :
 #RUNTIME_LAST					Last epoch time of execution - control var
 #COUNT_FAILURE_GEOIP	Failure count, control var
 #CLOCK_TOTAL          Total counter (database access)
 #The purpose of GeoIP faiulure countis to avoid timeouts in case the service is not available

 my $stmt = qq(CREATE TABLE IF NOT EXISTS tblStatus
             (RUNTIME_LAST INT PRIMARY KEY,
              COUNT_FAILURE_GEOIP INT,
              CLOCK_TOTAL INT
             ););
 my $ret = $dbh->do($stmt);
 if($ret < 0) {
   #print STDERR $DBI::errstr;
   syslog("Error Create Table IPV4 - $DBI::errstr");
 } else {
   #print STDERR "Table created successfully\n";
   #syslog("Create Table IPV4");
 }
}   #sub testSQLite


sub syslog {
 if ($flag_verbose) {
  print "$$\t$_[0]\n";
 } 
 if ($LoggingEnable == 1){
  open FH, ">>$Loggingpath" . '/' . "$appname.log" or die "Cannot append to $appname.log";
  my $tldate = strftime("%d-%m-%Y %H:%M:%S", localtime(time));
  print FH "$tldate\t$$\t$_[0]\n";
 } 
}  #sub syslog



sub incWebFailureRequest {
 #Increment the number of ailed web requests
 #Params:
 #-Counter
 
 my $inc = ($_[0] + 1);
 
 syslog("IP-To-Country/ASN - Connection Error, we increase the counter for today to: $inc");
 my $sqlstr = "UPDATE tblStatus set COUNT_FAILURE_GEOIP = $inc";
 syslog("$sqlstr");
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
}

sub getIPDetails {
 #Get IP Address details using 3rd party service
 #Params:
 #-IP Address
 #-Fail Counter
 #-Max Fail Counter 
 #Return:
 #-IP Country
 #-IP ASN

 my ($proto,$htmlcurip,$debstr,$tmpip,$tmpcountry,$json,$data);
 my $tmpasn="---"; #Set a placeholder for display purposes if no ASN is available
 my $tmpMaxCount=$_[2];
 #ToDo: Set this to configurationfile ( 3 )
 my $tmpFailedCnt=$_[1];

 #Get Param
 $htmlcurip=$_[0]; 			#get ip from param
 $htmlcurip=~s/:/\%3A/g;    #html encoding for IPv6


 if ($tmpFailedCnt >= $tmpMaxCount) { #more then 3 attemps per day will get refused
  syslog("IP-To-Country/ASN - Request: No more requests as too many errors attempted for today ($tmpFailedCnt/$tmpMaxCount)");
  return undef;
 } else {
  syslog("IP-To-Country/ASN - Request ($tmpFailedCnt/$tmpMaxCount):  $htmlcurip");
 }

 #Init Agent
 my $ua= LWP::UserAgent->new(timeout => 5);   #Init LWP, assign $ua
 $ua->agent('fail2nft-' . $version);          #Set agent name 


 #############
 #abstractapi
 #############
 if ($GeoIP_NAME =~ /abstractapi.com/i){         #Get matching config name   
  syslog("IP-To-Country/ASN - Provider: abstractapi.com");
  my $tmpGeoip_URL="https://ipgeolocation.abstractapi.com/v1/?api_key=$GeoIP_KEY&ip_address=$htmlcurip";
  my $req = GET $tmpGeoip_URL;
  my $res = $ua->request($req);
  if ($res->is_success) {
   $json = JSON->new;                     #Init a new json object as we expect to receive json
   $data = $json->decode($res->content);  #decode json to $data
   $tmpcountry=$data->{country_code};                              #get json country
   $tmpasn="AS".$data->{connection}->{autonomous_system_number};   #get json asn
   syslog("IP-To-Country/ASN - Received: $tmpcountry/$tmpasn");
   sleep 1;  #need to wait for 1 second due to the limitation of the free plan
   return $tmpcountry,$tmpasn;            
  } else {   #if ($res->is_success) 
   syslog("HTTP Error " . $res->status_line	 );
   &incWebFailureRequest($tmpFailedCnt);
   return undef;
  } 	     #else if ($res->is_success)
 }           #if ($GeoIP_NAME =~ /ip2loc.com/i)


 #########
 #ipapi.co
 #########
 if ($GeoIP_NAME =~ /ipapi.co/i){         #Get matching config name   
  syslog("IP-To-Country/ASN - Provider: ipapi.co");
  my $tmpGeoip_URL="https://ipapi.co/$htmlcurip/json";
  my $req = GET $tmpGeoip_URL;
  my $res = $ua->request($req);
  if ($res->is_success) {
   $json = JSON->new;                     #Init a new json object as we expect to receive json
   $data = $json->decode($res->content);  #decode json to $data
   $tmpcountry=$data->{country};          #get json country
   $tmpasn=$data->{asn};                  #get json asn
   syslog("IP-To-Country/ASN - Received: $tmpcountry/$tmpasn");
   return $tmpcountry,$tmpasn;            
  } else {   #if ($res->is_success) 
   syslog("HTTP Error " . $res->status_line	 );
   &incWebFailureRequest($tmpFailedCnt);
   return undef;
  } 	     #else if ($res->is_success)
 }           #if ($GeoIP_NAME =~ /ip2loc.com/i)
 


 #######
 #ip2loc
 #######
  if ($GeoIP_NAME =~ /ip2loc.com/i){         #Get matching config name   
  syslog("IP-To-Country/ASN - Provider: ip2loc.com (NO ASN SUPPORT)");
  my $tmpGeoip_URL="https://ipapi.co/$htmlcurip/json";
  my $tmpGeoip_URL="https://api.ip2loc.com/$GeoIP_KEY/$htmlcurip?include=country_alpha_2";
  my $req = GET $tmpGeoip_URL;
  my $res = $ua->request($req);
  if ($res->is_success) {
   $json = JSON->new;                     #Init a new json object as we expect to receive json
   $data = $json->decode($res->content);  #decode json to $data
   $tmpcountry=$data->{country_alpha_2};  #get json country
   syslog("IP-To-Country/ASN - Received: $tmpcountry/$tmpasn");
   return $tmpcountry,$tmpasn;            
  } else {   #if ($res->is_success) 
   syslog("HTTP Error " . $res->status_line	 );
   &incWebFailureRequest($tmpFailedCnt);
   return undef;
  } 	     #else if ($res->is_success)
 }           #if ($GeoIP_NAME =~ /ip2loc.com/i)



 
 #################
 #ipgeolocation.io
 #################
 if ($GeoIP_NAME =~ /ipgeolocation.io/i){         #Get matching config name   
  syslog("IP-To-Country/ASN - Provider: ipgeolocation.io (NO ASN SUPPORT)");
  my $tmpGeoip_URL="https://api.ipgeolocation.io/ipgeo?apiKey=$GeoIP_KEY&ip=$htmlcurip";
  my $req = GET $tmpGeoip_URL;
  my $res = $ua->request($req);
  if ($res->is_success) {
   $json = JSON->new;                     #Init a new json object as we expect to receive json
   $data = $json->decode($res->content);  #decode json to $data
   $tmpcountry=$data->{country_code2};    #get json country
   syslog("IP-To-Country/ASN - Received: $tmpcountry/$tmpasn");
   return $tmpcountry,$tmpasn;            
  } else {   #if ($res->is_success) 
   syslog("HTTP Error " . $res->status_line	 );
   &incWebFailureRequest($tmpFailedCnt);
   return undef;
  } 	     #else if ($res->is_success)
 }           #if ($GeoIP_NAME =~ /ip2loc.com/i)

  
 ############
 #coolgeo.org
 ############
 if ($GeoIP_NAME =~ /coolgeo.org/i){
  syslog("IP-To-Country/ASN - Provider: coolgeo.org");
  my $tmpGeoip_URL="https://xml.coolgeo.org/?myip=$htmlcurip";
  my $req = GET $tmpGeoip_URL;
  my $res = $ua->request($req);
  if ($res->is_success) {
   my $xml = new XML::Simple (KeyAttr=>[]);  # create XML object
   my $data = eval {$xml->XMLin($res->content,ForceArray => 0)};	 
   $tmpcountry=$data->{COUNTRY};   #get xml country     
   $tmpasn=$data->{ASN};           #get xml asn
   syslog("IP-To-Country/ASN - Received: $tmpcountry/$tmpasn");
   return $tmpcountry,$tmpasn;
  } else {   #if ($res->is_success) 
   syslog("HTTP Error " . $res->status_line	 );
   &incWebFailureRequest($tmpFailedCnt);
   return undef;
  }  #if ($res->is_success) 
 }   #if ($GeoIP_NAME =~ /coolgeo.org/i)


 if (!$GeoIP_URL) {
 	#Geosearch not configured OR Geosearch not configured properly
	syslog("IP-To-Country/ASN - Not Configured!");
 	return undef;
 }

}   #SUB getIPDetails




sub ReadConfig {
 #Get my XML config from $appname.xml	
 
 my $xml = new XML::Simple (KeyAttr=>[]); 
 my $data = $xml->XMLin("$_[0]",ForceArray => 0);
 $On_Success_Timer=$data->{Setup}->{On_Success_Timer};
 $On_Success_Renew=$data->{Setup}->{On_Success_Renew};
 $On_Fail_Timer=$data->{Setup}->{On_Fail_Timer};
 $On_Fail_Timer_Init=$data->{Setup}->{On_Fail_Timer};
 $Login_Fail_Counter=$data->{Setup}->{Login_Fail_Counter};
 $On_Fail_Double_Timer=$data->{Setup}->{On_Fail_Double_Timer};
 $Max_Reverse_Time=$data->{Setup}->{Max_Reverse_Time};
 $GeoIP_NAME=$data->{Setup}->{GeoIP_NAME}; #New with 1.3
 $GeoIP_KEY=$data->{Setup}->{GeoIP_KEY};   #New with 1.3
 $GeoIP_Connect_Failure_Max=$data->{Setup}->{GeoIP_Connect_Failure_Max};   #New with 1.3
 $GeoIP_URL=$data->{Setup}->{Geoip_URL};   #Capital Schema <1.3, not used anymore but kept due to compatibility reasons
 $Delete_Inactive_Records=$data->{Setup}->{Delete_Inactive_Records};
 $Reset_Record_Counter=$data->{Setup}->{Reset_Record_Counter};
 $Process_Timeout=$data->{Setup}->{Process_Timeout};
 $Discard_Private_IPAddress=$data->{Setup}->{Discard_Private_IPAddress};
 $LoggingEnable=$data->{Logging}->{Enable};
 $Loggingpath=$data->{Logging}->{Path};

 $Table_IPV4=$data->{NFTABLES}->{Table_IPV4};
 $Set_IPV4_drop=$data->{NFTABLES}->{Set_IPV4_drop};
 $Set_IPV4_accept=$data->{NFTABLES}->{Set_IPV4_accept};
 $Table_IPV6=$data->{NFTABLES}->{Table_IPV6};
 $Set_IPV6_drop=$data->{NFTABLES}->{Set_IPV6_drop};
 $Set_IPV6_accept=$data->{NFTABLES}->{Set_IPV6_accept};
 
 $EventMailLevel=$data->{Email}->{Level};
 $EventMailTo=$data->{Email}->{MailTo};
 $EventMailFrom=$data->{Email}->{MailFrom};
 $EventMailSMTP=$data->{Email}->{MailSMTP};
 $EventMailUser=$data->{Email}->{MailUser};
 $EventMailPassword=$data->{Email}->{MailPassword};
 
 $Set_vsftp=$data->{Logs}->{vsftp};
 $Set_mail=$data->{Logs}->{mail};
 $Set_grafana=$data->{Logs}->{grafana};
 $Set_splunk=$data->{Logs}->{splunk};
 #*************************************************************
 my $xml = new XML::Simple (KeyAttr=>[]); 
 my $data = $xml->XMLin($_[0],ForceArray => 1);

 for( @{$data->{Whitelist}}){
  #$syslog("Push: $_->{IP}");
  push(@whitelistip,$_->{IP});
 } 
 push(@whitelistip,'127.0.0.1');  #Set loopback for safety reasons
 push(@whitelistip,'::1');        #Set loopback for safety reasons

 for( @{$data->{Syslog}}){
  if ($_->{Enable}) {
   #print "Enable Slog: $_->{Enable} -- $_->{IP} ";
   push(@syslogip,$_->{IP});
  } 
 } 

 for( @{$data->{Country}}){
  if ($_->{Enable}) {
   #syslog("Read: $_->{Code}");
   $CountryCode_Enable_DoubleTimer{$_->{Code}}=$_->{On_Fail_Double_Timer};  #Enable doubling
   $CountryCode_Set_Timer{$_->{Code}}=$_->{On_Fail_Timer} ;           #Set timer per country
  } 
 } 

for( @{$data->{ASN}}){
  #syslog("Read: $_->{Code}");
  if ($_->{Enable}) { 
   #syslog("Read: $_->{Name}");
   $ASNName_Enable_DoubleTimer{$_->{Name}}=$_->{On_Fail_Double_Timer};  #Enable doubling
   $ASNName_Set_Timer{$_->{Name}}=$_->{On_Fail_Timer} ;           #Set timer per country
  } 
 } 


}  #ReadConfig




sub subCheckWhitelist {
	#Check if the given address is on our whitelist
	#Params
	#$_[0] - IP Address
	#Return: true/false
	foreach my $ip (@whitelistip) {
   if ($_[0] =~ /$ip/) {
    return 1;
   }
  }
  return undef;
}   #subCheckWhitelist



sub subTestConfiguration {
 #Params:
 #0 Name of Table
 #1 Name of Set
 #2 IP Protocol, this could be '4' of '6'
 #3 Reporting type, 1=Testing, 2=array of IPs
 #4 Dump configuration (if set then -3- does not matter
 #Return:
 #print "$_[0] - $_[1] - $_[2]\n";
 my ($flag_table,$flag_set,$flag_protocol_type,$pabbr,$strReturn);
 my $subRules = 0;
 my $ip_protocol;
 if ($_[2] == 4) {
  $ip_protocol='ipv4_addr';
  $pabbr='v4';
 } 
 if ($_[2] == 6) {
  $ip_protocol='ipv6_addr';
  $pabbr='v6';
 } 
 if (!$ip_protocol) {
 	print "No protocol given - abort\n";
 	exit;
 }
 #NFT Export to json does not work - we are using a workaround: #https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=926411
 my $nftStr=`nft -j list ruleset`;
 my $json = JSON->new;
 my $data = $json->decode($nftStr);
 if ($_[4] == 1) {
  return Dumper($data);
 }

 #print Dumper($data);

#SAMPLE DUMP:
#                            'set' => {
#                                       'flags' => [
#                                                    'timeout'
#                                                  ],
#                                       'elem' => [
#                                                   {
#                                                     'elem' => {
#                                                                 'timeout' => 60027,
#                                                                 'val' => '192.168.178.17',
#                                                                 'expires' => 59995
#                                                               }
#                                                   },
#                                                   {
#                                                     'elem' => {
#                                                                 'timeout' => 60027,
#                                                                 'expires' => 59990,
#                                                                 'val' => '192.168.178.18'
#                                                               }
#                                                   },
#                                                   {
#                                                     'elem' => {
#                                                                 'val' => '192.168.178.19',
#                                                                 'expires' => 2326,
#                                                                 'timeout' => 6027
#                                                               }
#                                                   }
#                                                 ],
#                                       'table' => 'filter_v4',
#                                       'handle' => 4,
#                                       'family' => 'ip',
#                                       'type' => 'ipv4_addr',
#                                       'name' => 'log2nft_drop'
#                                     }
#                          },
#                          {
#                            'set' => {
#                                       'table' => 'filter_v4',
#                                       'name' => 'log2nft_accept',
#                                       'type' => 'ipv4_addr',
#                                       'family' => 'ip',
#                                       'flags' => [
#                                                    'timeout'
#                                                  ],
#                                       'handle' => 5
#                                     }
#                          },


 for my $v (@{$data->{'nftables'}}){
  for my $w (keys (%{$v->{'set'}})){
 	 if ($v->{'set'}{'table'} eq $_[0]) {
 	 	$flag_table=1;
	  if ($v->{'set'}{'name'} eq $_[1]) {
	  	$flag_set=1;
	   if ($v->{'set'}{'type'} eq $ip_protocol) {
	   	$flag_protocol_type=1;
	    for my $s ( @{$v->{'set'}{'elem'}} ) {
	     $subRules++;
	     #print Dumper($s);
	     #print "======>>>>  $s->{'elem'}{'val'} \n";
	     #print "======>>>>  $s->{'elem'}{'timeout'} \n";
	     #print "======>>>>  $s->{'elem'}{'expires'} \n";
	     #for my $t (keys (%{$s->{'elem'}})){ ###Get the element as HASH ref and print
       # #print Dumper($t);
       #} 
       $hIPData{$s->{'elem'}{'val'}}{'timeout'}=$s->{'elem'}{'timeout'};
       #print "=====XXXXX>>> $hIPData{$s->{'elem'}{'val'}}{'timeout'}\n";
       $hIPData{$s->{'elem'}{'val'}}{'expires'}=$s->{'elem'}{'expires'};
      }
      last;
	   }
	  }
	 }
   if (($flag_protocol_type) and ($flag_table) and ($flag_set)) { last }
  }
  if (($flag_protocol_type) and ($flag_table) and ($flag_set)) { last }
 } 

 if (!$flag_table) {
 	#print "($pabbr) Table\t\tFAIL\t=> $_[0]\n";
 	$strReturn.="($pabbr) Table\t\tFAIL\t=> $_[0]\n";
 } else {
 	#print "($pabbr) Table\t\tOK \t=> $_[0]\n";
 	$strReturn.= "($pabbr) Table\t\tOK \t=> $_[0]\n";
 }
 if (!$flag_set) {
 	#print "($pabbr) Named Set\t\tFAIL\t=> $_[1]\n";
 	$strReturn.="($pabbr) Named Set\t\tFAIL\t=> $_[1]\n";
 } else {	
 	#print "($pabbr) Named Set\t\tOK \t=> $_[1]\n";
 	$strReturn.="($pabbr) Named Set\t\tOK \t=> $_[1]\n";;
 }
 if (!$flag_protocol_type) {
 	#print "($pabbr) Protol Type\tFAIL \t=> $ip_protocol\n";
 	$strReturn.="($pabbr) Protol Type\tFAIL \t=> $ip_protocol\n";
 } else {
 	#print "($pabbr) Protol Type\tOK \t=> $ip_protocol\n";
 	$strReturn.="($pabbr) Protol Type\tOK \t=> $ip_protocol\n";
 }


 if ($_[3] eq 1) {
  if (($flag_table) and ($flag_set) and ($flag_protocol_type) ) {
	 #print "Test result:\t\tOK\n";
	 $strReturn.="($pabbr) Records:\t\t\t$subRules\n";
	 $strReturn.="Test result:\t\tOK\n";
	 return $strReturn;
  } else {
	 #print "Test result:\t\tError\n";
	 $strReturn="Test result:\t\tError\n";
	 #print Dumper($data);
	 return $strReturn;
  }
 } 
 if ($_[3] eq 2) {
  #print "ret\n";
  #print Dumper(%hIPData);
  #return (\%hIPData); 
  #return %hIPData;
 
  #return (\%hIPData);
  #return (\%hash1, \%hash2);
  #print Dumper(%hIPData);
 }
}   #sub subTestConfiguration



sub print_help {
	print"\nOne or more parameter is expected\n\n";
	print "-add\t\tAdd an IP Address to NFT, this option requires more options, samples:\n";
	print "-add -a -ip (ip address) -time (hours) - Add IP Address to the accept target\n";
	print "-add -d -ip (ip address) -time (hours) - Add IP Address to the drop target\n";
	print "-c\t\tCheck the logs, if specified then no more paramterets must be given\n";
	print "-delete\t\tDelete an IP address from the fail2nft database and NFT itself\n";
    print "-h\t\tPrint help\n";
	print "-i\t\tInitialize nft (typically used after a nft reset or server reboot)\n";
	print "-l -d\t\tList all known IP's out of our database, list drop rules only\n";
	print "-l\t\tList all known IP's out of our database\n";
	print "-l -a\t\tList all known IP's out of our database, list accepted rules only\n";
	print "-s\t\tCheck/create the sqlite database and exit\n";
	print "-t\t\tTest avaialble nft rulesets\n";
	print "-t -json\tPrint the NFT configuration using the json export\n";
	print "-testmail\tSend a testmail\n";
	print "-v\t\tBe verbose\n";
	print "-version\tprint version only\n";
	print  "\n--------------------------------------------------------\n";
	print "Samples:\n";
	print "-Check logs and apply to nft if needed (crontab/console):\n";
	print "# fail2nft -c\n";
	print "-Check logs and be verbose:\n";
	print "# fail2nft -c -i\n";
	print  "--------------------------------------------------------\n";
	print "Version:\t$version \n";
	print "Configuration:\t$configuration \n";
	print "Database:\t$databasepath\n\n";
	
}   #sub print_help

sub subListData {
 #List current sqlite recrods out of tblIP	
 #Global Params: 
 #$flag_accept - Print only accepted records
 #$flag_dropt -  Print only denied records
 #Return: None
 
 my ($scnt,$tlTE,$tAllow,$tNFT,$splprefix);
 if ($flag_accept) {
 	$splprefix=" where FLAG_WHITE=1 and APPLIED_TO_NFT=1";
 }

 if ($flag_drop) {
 	$splprefix=" where FLAG_WHITE=0 and APPLIED_TO_NFT=1";
 }

 my ($TimeLastExec,$GeoConnectionFailure,$ClockTotal)=&subGetStatus; #Get the previous run time data so we can search for a time delta
 my $sqlstr = "SELECT IP,FLAG_WHITE,TIME_FIRST_SEEN,TIME_LAST_SEEN,TIME_EXPIRES,APPLIED_TO_NFT,COUNTER_CUR,COUNTRY,ASN,COUNTER_NFT,LOG FROM tblIP" . $splprefix . " order by TIME_LAST_SEEN";
 my $tldate = strftime("%Y-%m-%d %H:%M:%S", localtime($TimeLastExec));
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
 print "\nlast refresh: $tldate\n\n";
 print    "Count\tAccept\tDeny\tCCNT\tTCNT\tIP\t\tFirst Seen\t\tLast Seen\t\tTime Expires\t\tCountry/ASN/LOG\n";
 print "-----\t------\t----\t---\t---\t--------------\t----------------\t----------------\t-------------\t\t---------------\n";
 while (my ($sqlIP,$slqFlagWhite,$sqlTIME_FIRST_SEEN,$sqlTIME_LAST_SEEN,$sqlTIME_EXPIRES,$sqlAPPLIED_TO_NFT,$sqlCOUNTER_CUR,$sqlCOUNTRY,$sqlASN,$sqlCOUNTER_NFT,$sqlLOG)=$sql->fetchrow_array) {
 	$scnt++;
  my $tlFS = strftime("%Y-%m-%d %H:%M", localtime($sqlTIME_FIRST_SEEN));
  my $tlLS = strftime("%Y-%m-%d %H:%M", localtime($sqlTIME_LAST_SEEN));
  if ($sqlTIME_EXPIRES == 0) {
  	$tlTE = "----------------"
  } else {	
    $tlTE = strftime("%Y-%m-%d %H:%M", localtime($sqlTIME_EXPIRES));
  } 

  if ($sqlAPPLIED_TO_NFT ne "1") {
   $tAllow="-";
   $sqlAPPLIED_TO_NFT="-";
  } else {
   $tAllow=$slqFlagWhite;
   #$sqlAPPLIED_TO_NFT="-";
  }
  
  if ($tAllow) {
 	 if ($sqlAPPLIED_TO_NFT eq "1") {
    $tAllow="1";
    $sqlAPPLIED_TO_NFT="-";
   }
  }
  
  if ($sqlIP=~/:/) {
   $tlFS="\t";  #don't display item if ip is v6
  } 
  print "$scnt\t$tAllow\t$sqlAPPLIED_TO_NFT\t$sqlCOUNTER_CUR/$Login_Fail_Counter\t$sqlCOUNTER_NFT\t$sqlIP\t$tlFS\t$tlLS\t$tlTE\t$sqlCOUNTRY/$sqlASN/$sqlLOG\n";
 } 
 print "\n";
}  #subListData



sub SendToSyslog {
	#Send data to the syslog server
	#Params: The Message
	#Return: None
	my $s;
	foreach (@syslogip){
   $s=new Net::Syslog(Facility=>'local4',Priority=>'debug',SyslogHost=>$_);
   print "SEND $_[0] $_\n";
   $s->send($_[0],Priority=>'info');
  } 
}   #SendToSyslog


sub subInitNFT {
 #Initialize NFT with the given values from SQLite, we use the applied to NFT values and observe the expiry time and relation.
 #Please note that this process does not cross check the given NFT settings, it simply applies the commands
 #Params: None
 #Return: None
 my $sqlstr = "SELECT IP,TIME_EXPIRES,FLAG_WHITE,COUNTRY,ASN FROM tblIP where APPLIED_TO_NFT=1";
 my ($tmpSET,$tmpTimer,$cmdstat);
 $tmpSET=undef;
 $tmpNFTNameTable=undef; #Reset previous stats to be secure
 #print "$sqlstr\n";
 my $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
 while (my ($sqlIP,$sqlTIME_EXPIRES,$sqlFLAG_WHITE,$detailIP_Country,$detailIP_ASN)=$sql->fetchrow_array) {
 	#print "$sqlIP,$sqlTIME_EXPIRES,$sqlFLAG_WHITE,$detailIP_Country,$detailIP_ASN\n";
  ($tmpIPVer,$tmpIPType) = subGetIPInfo($sqlIP);
  if ($sqlFLAG_WHITE == 1) {
   if ($tmpIPVer == 4) {
	  if ((!$Table_IPV4) or (!$Set_IPV4_drop) or (!$Set_IPV4_accept) ){
	 	 syslog("WARNING: IP4 is not configured - abort");
	  }
    $tmpSET=$Set_IPV4_accept;
    $tmpNFTNameTable=$Table_IPV4;
   } 
   if ($tmpIPVer == 6) {
	  if ((!$Table_IPV6) or (!$Set_IPV6_drop) or (!$Set_IPV6_accept) ){
	 	 syslog("WARNING: IP6 is not configured - abort");
	 	}
    $tmpSET=$Set_IPV6_accept;
    $tmpNFTNameTable=$Table_IPV6;
   } 
   $tmpTimer=($sqlTIME_EXPIRES-time);
  } else {
   if ($tmpIPVer == 4) {
	  if ((!$Table_IPV4) or (!$Set_IPV4_drop) or (!$Set_IPV4_accept) ){
	 	 syslog("WARNING: IP4 is not configured - abort");
	  }
    $tmpSET=$Set_IPV4_drop;
    $tmpNFTNameTable=$Table_IPV4;
   } 
   if ($tmpIPVer == 6) {
    if ((!$Table_IPV6) or (!$Set_IPV6_drop) or (!$Set_IPV6_accept) ){
	 	 syslog("WARNING: IP6 is not configured - abort");
	  }
    $tmpSET=$Set_IPV6_drop;
    $tmpNFTNameTable=$Table_IPV6;
   } 
   $tmpTimer=($sqlTIME_EXPIRES-time);
  }
  if ($tmpTimer > 0) {
   $cmdstat=subApplyNFT($sqlIP,$tmpNFTNameTable,$tmpSET,$tmpTimer,$detailIP_Country,$detailIP_ASN,$tmpIPVer); #Apply NFT Rule
   
	 if (!$cmdstat) {
	  syslog("INIT: NFT applied OK 1");
	 } else {
	  	syslog("INIT: NFT reports an error, see /tmp/$appname"."_error.log ");
	 }
	} 
 }
 syslog("Initialization finished");
}  #sub


sub subDeleteIP {
 #Delete IP address out of our database AND delete the ip out of the NFT set 
 #Params:
 #1: IP
 #Return: None
 my ($tmpTable,$tmpSet,$tmpNFTTarget,$tmpIPType);
 #print "Delete $_[0]\n";
 
 $tmpNFTNameTable=undef;  #Reset previous stats to be secure
 $tmpNFTTarget=undef;     #Reset previous stats to be secure
 my $sqlstr = "SELECT APPLIED_TO_NFT,FLAG_WHITE FROM tblIP where IP='$_[0]'";
 my $sql = $dbh->prepare($sqlstr);
 #print "$sqlstr\n";
 $sql->execute; 
 #(my ($sqlRUNTIME_LAST,$sqlRUNTIME_GEOIP,$sqlCOUNT_FAILURE_GEOIP)=$sql->fetchrow_array);
 (my ( $sqlAPPLIED_TO_NFT, $sqlFLAG_WHITE)=$sql->fetchrow_array);
 #print "$sqlAPPLIED_TO_NFT, $sqlFLAG_WHITE\n";

 ($tmpIPVer,$tmpIPType) = subGetIPInfo($_[0]); #Get IP Verison and Type
 if ($tmpIPVer == 4) { #Map IP Version to NFTable
	$tmpTable=$Table_IPV4;
	if ($sqlFLAG_WHITE == 1) {
	 $tmpNFTTarget=$Set_IPV4_accept;
	} else {
	 $tmpNFTTarget=$Set_IPV4_drop;
  }	
 }	
 if ($tmpIPVer == 6) { #Map IP Version to NFTable
	$tmpTable=$Table_IPV6;
 	if ($sqlFLAG_WHITE == 1) {
	 $tmpNFTTarget=$Set_IPV6_accept;
	} else {
	 $tmpNFTTarget=$Set_IPV6_drop;
  }	

 }	

 if ($sqlAPPLIED_TO_NFT == 1 ) {
  my $cmdstat=subDeleteFromNFT($_[0],$tmpTable,$tmpNFTTarget,$tmpIPVer); #Apply Delete NFT Rule
  if (!$cmdstat) {
   ##syslog("NFT applied OK 1");
   #$sqlstr = "update tblIP set APPLIED_TO_NFT=0,TIME_EXPIRES=0,PREVIOUS_LOCK_TIMER=0,COUNTER_CUR=0 where IP='$_[0]'";
   ##print "$sqlstr\n";
   #$sql = $dbh->prepare($sqlstr);
   #$sql->execute; 
  } else {
   syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
  }
 } else {
   syslog("NFT: Unknown IP: $_[0]");
 }

 $sqlstr = "delete from tblIP where IP='$_[0]'";
 $sql = $dbh->prepare($sqlstr);
 $sql->execute; 
 syslog("SQL: $sqlstr");


}   #subDeleteIP


sub subDeleteFromNFT {
 #Params
 #0: IP
 #1: Name of Table 
 #2: Name of Set (differ between Block/Allow)
 #3: IP Version
 #Return: 
 # Def = error
 # Undef = OK
 #Sample: nft delete element ip filter_v4 log2nft_accept \{ 192.168.178.19\}
 my ($sub_nftStr,$errstr,$strComment,$tmpVer);
 #print "$_[0] - $_[1] - $_[2] - $_[3]\n"	;
 if ($_[4]) {
 	$strComment="$_[4]";	
 }
 if ($_[5]) {
 	$strComment.="/$_[5]";	
 }
 if ($_[3] == 4) {	
	$tmpVer="ip";
 }
 if ($_[3] == 6) {	
	$tmpVer="ip6";
 }
 
 $sub_nftStr="/usr/sbin/nft delete element $tmpVer " .$_[1] . ' ' . $_[2] . ' \{'. $_[0] .'\}';
 syslog("NFT: $sub_nftStr ");
 
 my $output = system("$sub_nftStr 2>>/tmp/$appname"."_error.log"); #Print the error message(s) to /tmp
 if ($output ne 0){
  	return 1;
 } 
 return undef;
}  # subDeleteFromNFT

sub subCheckForDayChange {
 #Detect if a day change occured between the epoch date
 #Param:
 #0: Epoch to check
 #Return
 #Defined if a day change is in between

 my $curday=strftime("%d", localtime(time));
 my $actday=strftime("%d", localtime($_[0]));

 if ($curday ne $actday) {
   #print "Day change - $curday $actday\n";
   return 1;
 }
 return undef;
}

sub subCheckForMonthChange {
 #Detectif a month change is in between the epoch date
 #Param:
 #0: Epoch to check
 #Return
 #Defined if a month change is in between

 my $curmonth=strftime("%m", localtime(time));
 my $actmonth=strftime("%m", localtime($_[0]));

 if ($curmonth ne $actmonth) {
   #print "Month change - $curmonth $actmonth\n";
   return 1;
 }
 return undef;
}


sub subDailyMaintenance {
  #########################
  #Delete Expired Records
  #########################
  #SearchTime=time-Delete_Inactive_Records
  #SELECT * FROM tblIP where TIME_LAST_SEEN < 1570729398 and APPLIED_TO_NFT=0
 
  my (@deleteIP,@resetIP,$cnt,$sqlstr,$sql);	
  syslog("Apply Daily Maintenance");
  $sqlstr = "UPDATE tblStatus set COUNT_FAILURE_GEOIP = 0";
  syslog("SQL: $sqlstr");
  $sql = $dbh->prepare($sqlstr);
  $sql->execute; 

  
  $sqlstr = "Update tblIP set TIME_EXPIRES=0,COUNTER_CUR=0 where APPLIED_TO_NFT = 0 and TIME_EXPIRES < " .time  . " and TIME_EXPIRES>0;";
  syslog("SQL: $sqlstr");
  my $sql = $dbh->prepare($sqlstr);
  $sql->execute; #Fire and forget ...

  
	my $searchtime = (time - $Delete_Inactive_Records) ;
	syslog("SQL: (PREPARE DELETE): SELECT * FROM tblIP where TIME_LAST_SEEN < $searchtime and APPLIED_TO_NFT=0"); 
  $sqlstr = "SELECT IP,PREVIOUS_LOCK_TIMER,TIME_LAST_SEEN FROM tblIP where TIME_LAST_SEEN < $searchtime and APPLIED_TO_NFT=0";
  $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
  while (my ($sqlIP,$sqlPREVIOUS_LOCK_TIMER,$sqlTIME_LAST_SEEN)=$sql->fetchrow_array) {
   #syslog ("BETA DELETE $sqlIP - $sqlPREVIOUS_LOCK_TIMER,$sqlTIME_LAST_SEEN");
   if ($searchtime > ($sqlTIME_LAST_SEEN + $sqlPREVIOUS_LOCK_TIMER)) {
   my $tl11 = strftime("%Y-%m-%d %H:%M:%S", localtime($sqlTIME_LAST_SEEN));
   my $tl12 = strftime("%Y-%m-%d %H:%M:%S", localtime($sqlTIME_LAST_SEEN + $sqlPREVIOUS_LOCK_TIMER));
    #syslog ("BETA FINAL DELETE $sqlIP - $sqlPREVIOUS_LOCK_TIMER,$sqlTIME_LAST_SEEN ---- $tl11 PREVCLOCK: $tl12");
    push (@deleteIP,$sqlIP);
   }
  }
  foreach (@deleteIP) {
 	 $sqlstr = "Delete from tblIP where IP='$_';";
 	 syslog("SQL $sqlstr");
   $sql = $dbh->prepare($sqlstr);
   $sql->execute; 
  }

  #######################
  #Reset Inactive Counter
  #######################

  
  $searchtime = (time - $Reset_Record_Counter) ;
  syslog("SQL: (PREPARE RESET): SELECT * FROM tblIP where TIME_LAST_SEEN < $searchtime and APPLIED_TO_NFT=0 and Counter_CUR>0"); 
  $sqlstr = "SELECT IP,PREVIOUS_LOCK_TIMER,TIME_LAST_SEEN FROM tblIP where TIME_LAST_SEEN < $searchtime and APPLIED_TO_NFT=0 and Counter_CUR>0";
  $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
  while (my ($sqlIP,$sqlPREVIOUS_LOCK_TIMER,$sqlTIME_LAST_SEEN)=$sql->fetchrow_array) {
    #syslog ("BETA RESET $sqlIP - $sqlPREVIOUS_LOCK_TIMER,$sqlTIME_LAST_SEEN");
    if ($searchtime > ($sqlTIME_LAST_SEEN + $sqlPREVIOUS_LOCK_TIMER)) {
     my $tl11 = strftime("%Y-%m-%d %H:%M:%S", localtime($sqlTIME_LAST_SEEN));
     my $tl12 = strftime("%Y-%m-%d %H:%M:%S", localtime($sqlTIME_LAST_SEEN + $sqlPREVIOUS_LOCK_TIMER));
     #syslog ("BETA FINAL RESET $sqlIP - $sqlPREVIOUS_LOCK_TIMER,$sqlTIME_LAST_SEEN ---- $tl11 PREVCLOCK: $tl12");
     push (@resetIP,$sqlIP);
   }
  }
  $cnt=0;
  foreach (@resetIP) {
  	$cnt++;
 	#syslog("$cnt FINAL Maintenance: Reset IP $_");
 	$sqlstr = "Update tblIP SET Counter_CUR=0 where IP='$_'";
 	syslog("SQL: $sqlstr");
    $sql = $dbh->prepare($sqlstr);
    $sql->execute; 
  }
}   #Sub




sub GetProcessInfo {
 #Print the process PID and time for how long the process is running
 #Params
 #0 process name
 #1 my own PID
 #Return
 #-PID
 #-Time running in seconds
 my ($t,$p,$f);
 $t = new Proc::ProcessTable;
 foreach my $p ( @{$t->table} ){
  if ($p->{'pid'} ne $_[1]){
   if ($p->{'fname'} =~ /$_[0]/i){
    foreach $f ($t->fields){
     if ($f =~ /start/){
      return (($p->{'pid'},(time-$p->{$f})));
     }  #if ($f =~ /start/)
    }   #foreach $f ($t->fields)
   }    #if ($p->{'fname'} =~ /$_[0]/i)
  }
 }     #foreach my $p ( @{$t->table} )
 return undef;
}      #sub



sub MessageViaEmail {
 #Param:
 #-0 Success/Fail
 #-1 IP
 #-2 TImer
 #-3 Country
 #-4 ASN
 #-5 IP Ver
 #-6 Log info / proto
 my ($strSubject,$strBody);
 if (($_[0] == 1) and (($EventMailLevel==1) or ($EventMailLevel==3))) { #Success message
 	 syslog("Send success mail message");
 	 $strSubject="Successful login from $hostname";
 	 $strBody="
 	  IP=$_[1]
 	  Country=$_[3]
 	  ASN=$_[4]
 	  Timer=$_[2]
 	  Ver=$_[5]
 	  Log=$_[6]
 	 ";
 	 &Sendmail($strSubject,$strBody);
 }

 if (($_[0] == 2) and (($EventMailLevel==2) or ($EventMailLevel==3))) { #Failed message
 	 syslog("Send failure mail message");
 	 $strSubject="Failed login from $hostname";
 	 $strBody="
 	  IP=$_[1]
 	  Country=$_[3]
 	  ASN=$_[4]
 	  Timer=$_[2]
 	  Ver=$_[5]
 	  Log=$_[6]
 	 ";
 	 &Sendmail($strSubject,$strBody);
 }
 
}


sub TestEmail {
 #Param:
 #-0 Success/Fail
 #-1 IP
 #-2 TImer
 #-3 Country
 #-4 ASN
 #-5 IP Ver
 #-6 Log info / proto
 my ($strSubject,$strBody);
 syslog("Testing email");
 if (($_[0] == 1) or (($EventMailLevel==1) or ($EventMailLevel==3))) { #Success message
 	 $strSubject="Test Subject";
 	 $strBody="Test";
 	 &Sendmail($strSubject,$strBody);
 } else {
  syslog("Email option is not configured");	
 }
}

sub Sendmail {
 #Send a simple email
 #Param:
 #1 Subject
 #2 Body
 #Return: Nothing
 my $EventMailMethod="PLAIN LOGIN";
 my $tldate = strftime("%Y-%m-%d %H:%M:%S", localtime(time));
 my %mail = (
    To      => $EventMailTo,
    From    => $EventMailFrom,
    Subject => $_[0],
    Message => $_[1],
    smtp  => $EventMailSMTP
 );
 if (($EventMailUser) and ($EventMailPassword)) {
  $mail{auth} = {user=>$EventMailUser, password=>$EventMailPassword, method=>$EventMailMethod, required=>1 };
 } 
 sendmail(%mail) or syslog($Mail::Sendmail::error);
 syslog("OK. Log says: " . $Mail::Sendmail::log) ;
}



sub subReset {
	my $sqlstr = "delete FROM tblIp";
  my $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
  syslog("Data has been deleted from tblIP");
}


sub MessageViaSyslog {
 my ($sip,$syslogIP);
 if ($usesyslog) {
  foreach $syslogIP (@syslogip) {
   syslog("Send to Syslog Server: $syslogIP - MSG: $_[0]");
   $sip=new Net::Syslog(Facility=>'local4',Priority=>'debug',SyslogHost=>$syslogIP);
   $sip->send($_[0],Priority=>'info');
  }
 } else {
 	syslog("Warning: cannot use Net::Syslog, check installation and configuration");
 } 
} 


sub UpdateIPAllow {
 #Reassign already given accepted IP
 #Update IP 
 #Params:
 #0: IP
 #1: Country
 #2: ASN
 #3: Loginfo
 #4 Counter_tot
 #5 counter_nft
 #Return: None
 syslog("Reassining previous allowed address: $_[0]" );
 my ($tmpTable,$tmpSet,$tmpNFTTarget,$tmpIPType,$sqlstr,$sql);
 
 ($tmpIPVer,$tmpIPType) = subGetIPInfo($_[0]); #Get IP Verison and Type 
 if ($tmpIPVer == 4) { #Map IP Version to NFTable
	$tmpTable=$Table_IPV4;
  $tmpNFTTarget=$Set_IPV4_accept;
 }	
 if ($tmpIPVer == 6) { #Map IP Version to NFTable
	$tmpTable=$Table_IPV6;
	$tmpNFTTarget=$Set_IPV6_accept;
 }	

 my $cmdstat=subDeleteFromNFT($_[0],$tmpTable,$tmpNFTTarget,$tmpIPVer); #Apply Delete NFT Rule
 if (!$cmdstat) {
	syslog("NFT delete applied OK 1"); #OK!
 } else {
  syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
 }


 my $cmdstat=subApplyNFT($_[0],$tmpTable,$tmpNFTTarget,$On_Success_Timer,$_[2],$_[3],$tmpIPVer,$_[4]); #Apply NFT Rule
 #Should we really do this? Think of video streaming ???
 #&MessageViaEmail(1,$_[0],$On_Success_Timer,$_[1],$_[2],$tmpIPVer,$_[3]);
 #&MessageViaSyslog("M=F2N LOCK=0 IP=$_[0] TIMER=$On_Success_Timer COUNTRY=$_[1] ASN=$_[2] LOG=$_[3]");
 if (!$cmdstat) {
  syslog("NFT applied OK 1");
  my $newTime= $On_Success_Timer + time;
  $sqlstr = "update tblIP set TIME_EXPIRES=$newTime where IP='$_[0]'";
  syslog("Update already accepted record: $sqlstr");
  $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
 } else {
  syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
 }
}

sub subAddIP {
	#Add a IP address to NFT and SQL via console
	#0: Target (a/d)
	#1: IP
	#2: Time in hours
	#Return None
	
	my ($tmpTarget,$tmpIPExist,$sqlstr,$sql);
  ($tmpIPVer,$tmpIPType) = subGetIPInfo($_[1]); #Get IP Verison and Type
	
	
	if ($tmpIPType eq "PRIVATE") {
		print "IP Address $_[1] is private and cannot be added\n";
		exit;
	}

	if ($tmpIPType eq "LOOPBACK") {
		print "IP Address $_[1] is loopback and cannot be added\n";
		exit;
	}

	if ($tmpIPType eq "LINK-LOCAL-UNICAST") {
		print "IP Address $_[1] is link-local and cannot be added\n";
		exit;
	}
	
  if ($tmpIPVer == 4) { #Map IP Version to NFTable
   if ((!$Table_IPV4) or (!$Set_IPV4_drop) or (!$Set_IPV4_accept) ){
	 	 syslog("WARNING: IP4 is not configured - abort");
   }
	 $tmpNFTNameTable=$Table_IPV4;
	 if ($_[0] eq "a") {
	  $tmpNFTTarget=$Set_IPV4_accept;
	 } 
	 if ($_[0] eq "d") { 
	 	$tmpNFTTarget=$Set_IPV4_drop;
	 }	
  }	  #if ($tmpIPVer == 4) { #Map IP Version to NFTable
  if ($tmpIPVer == 6) { #Map IP Version to NFTable
   if ((!$Table_IPV6) or (!$Set_IPV6_drop) or (!$Set_IPV6_accept) ){
	 	 syslog("WARNING: IP6 is not configured - abort");
   }
	 $tmpNFTNameTable=$Table_IPV6;
	 if ($_[0] eq "a") {
	  $tmpNFTTarget=$Set_IPV6_accept;
	 } 
	 if ($_[0] eq "d") { 
	 	$tmpNFTTarget=$Set_IPV6_drop;
	 }	
  }	#  if ($tmpIPVer == 6) { #Map IP Version to NFTable


  $sqlstr = "SELECT IP,APPLIED_TO_NFT from tblIP where IP='$_[1]';" ;
  $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
  (my ($sqlIP,$sqlAPPLIED_TO_NFT)=$sql->fetchrow_array); 
	if ($sqlAPPLIED_TO_NFT) {
   print "Cannot add IP Address $_[1] because it already applied to nft\n";
	 exit;
  }

  $sqlstr = "SELECT IP from tblIP where IP='$_[1]';" ;
  $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
  (my ($sqlIP)=$sql->fetchrow_array); 
	if ($sqlIP) {
	 $tmpIPExist=1
	}





  my $tmpTimer=($_[2] * 60 *60);
  if ($_[0] eq "a") {
   $tmpTarget="1"
  } 
  if ($_[0] eq "d") {
   $tmpTarget="0"
  } 
  
  #All checks are done - now fianly apply  
  my $cmdstat=subApplyNFT($_[1],$tmpNFTNameTable,$tmpNFTTarget,$tmpTimer,"x","x",$tmpIPVer,"Manual"); #Apply NFT Rule
  if (!$cmdstat) {
   syslog("NFT applied OK 1");
   if (!$tmpIPExist)  {
    &addtblIP($_[1],"x","x",$tmpTarget,$tmpTimer,1,'0','1','Manual');  
   } else {
    &subUpdateIP($_[1],"-","-",0,$tmpTimer,1,0);  
   } 
  } else {
   syslog("NFT reports an error, see /tmp/$appname"."_error.log ");
  }
	
}  #sub subAddIP 


sub subGetIPInfo(){ # IP => Number
 #Return the type of IP Version, either 4 or 6 or undef for invalid strings
 #Param: IP Address
 #Return IP Type: 4 or 6
 if ($_[0]) {  #1.1
  my $ip = new Net::IP ($_[0]) or print "Error reading " . (Net::IP::Error());
  return ($ip->version(),$ip->iptype());
 } else {
  return undef;
 }
}

sub subMonthlyMaintenance {
  my $sqlstr = "VACUUM" ;
  syslog("Monthly maintenance - SQL: $sqlstr");
  my $sql = $dbh->prepare($sqlstr);
  $sql->execute; 
}
